package com.ruoyi.school.paper.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ruoyi.common.core.domain.PageQuery;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.utils.JsonUtils;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.redis.RedisUtils;
import com.ruoyi.school.paper.domain.DbQuestionBank;
import com.ruoyi.school.paper.domain.DbQuestionBankAnswer;
import com.ruoyi.school.paper.domain.bo.DbQuestionBankAnswerBo;
import com.ruoyi.school.paper.domain.bo.DbQuestionBankBo;
import com.ruoyi.school.paper.domain.bo.QuestionBankAndAnswerBo;
import com.ruoyi.school.paper.domain.vo.DbQuestionBankAndAnswerVo;
import com.ruoyi.school.paper.domain.vo.DbQuestionBankVo;
import com.ruoyi.school.paper.mapper.DbQuestionBankAnswerMapper;
import com.ruoyi.school.paper.mapper.DbQuestionBankMapper;
import com.ruoyi.school.paper.service.IDbQuestionBankService;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * 题库主Service业务层处理
 *
 * @author zyf
 * @date 2023-07-20
 */
@RequiredArgsConstructor
@Service
public class DbQuestionBankServiceImpl implements IDbQuestionBankService {

    private final DbQuestionBankMapper baseMapper;

    private final DbQuestionBankAnswerMapper questionBankAnswerMapper;

    private final String DEFAULT_QUESTIOBN_BANK_KEY = "question_bank:default_question_bank";

    /**
     * 查询题库主
     */
    @Override
    public DbQuestionBankVo queryById(Long id) {
        return baseMapper.selectVoById(id);
    }

    @Override
    public DbQuestionBankAndAnswerVo getById(Long id) {
        return baseMapper.selectDetailById(id);
    }

    /**
     * 查询题库主列表
     */
    @Override
    public TableDataInfo<DbQuestionBankVo> queryPageList(DbQuestionBankBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<DbQuestionBank> lqw = buildQueryWrapper(bo);
        Page<DbQuestionBankVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询题库主列表
     */
    @Override
    public List<DbQuestionBankVo> queryList(DbQuestionBankBo bo) {
        LambdaQueryWrapper<DbQuestionBank> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<DbQuestionBank> buildQueryWrapper(DbQuestionBankBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<DbQuestionBank> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getCategoryId() != null, DbQuestionBank::getCategoryId, bo.getCategoryId());
        lqw.eq(bo.getSpecies() != null, DbQuestionBank::getSpecies, bo.getSpecies());
        lqw.eq(bo.getPhaseId() != null, DbQuestionBank::getPhaseId, bo.getPhaseId());
        lqw.eq(bo.getType() != null, DbQuestionBank::getType, bo.getType());
        lqw.eq(bo.getIntensity() != null, DbQuestionBank::getIntensity, bo.getIntensity());
        lqw.eq(StringUtils.isNotBlank(bo.getTopic()), DbQuestionBank::getTopic, bo.getTopic());
        lqw.eq(StringUtils.isNotBlank(bo.getRightAnswers()), DbQuestionBank::getRightAnswers, bo.getRightAnswers());
        lqw.eq(StringUtils.isNotBlank(bo.getRightAnswersPic()), DbQuestionBank::getRightAnswersPic, bo.getRightAnswersPic());
        lqw.eq(StringUtils.isNotBlank(bo.getResolveCourse()), DbQuestionBank::getResolveCourse, bo.getResolveCourse());
        lqw.eq(StringUtils.isNotBlank(bo.getResolveVideo()), DbQuestionBank::getResolveVideo, bo.getResolveVideo());
        lqw.eq(bo.getStatus() != null, DbQuestionBank::getStatus, bo.getStatus());
        return lqw;
    }

    /**
     * 新增题库主
     */
    @Override
    public Boolean insertByBo(DbQuestionBankBo bo) {
        DbQuestionBank add = BeanUtil.toBean(bo, DbQuestionBank.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 新增题库
     * <p>若为选择题则将相关选项加入 db_question_bank_answer表中
     *
     * @param bo 题目参数及答案列表
     * @return 操作结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean insertByBo(QuestionBankAndAnswerBo bo) {
        DbQuestionBank insertRecord = BeanUtil.toBean(bo, DbQuestionBank.class);
        validEntityBeforeSave(insertRecord);
        boolean questionInsertFlag = baseMapper.insert(insertRecord) > 0;
        // 若为选择题则同步插入至选项答案中
        boolean answerInsertFlag = true;
        if (questionInsertFlag) {
            // 将新增的题目添加到redis题库缓存中
            List<DbQuestionBank> questionBanks = JsonUtils.parseArray(RedisUtils.getCacheObject(DEFAULT_QUESTIOBN_BANK_KEY), DbQuestionBank.class);
            questionBanks.add(insertRecord);
            RedisUtils.setCacheObject(DEFAULT_QUESTIOBN_BANK_KEY, JsonUtils.toJsonString(questionBanks));

            // 判断是否选择题，是则添加选择题选项
            if (insertRecord.getSpecies() == 2 && CollUtil.isNotEmpty(bo.getOptions())) {
                List<DbQuestionBankAnswer> options = new ArrayList<>(bo.getOptions().size());
                for (DbQuestionBankAnswerBo option : bo.getOptions()) {
                    DbQuestionBankAnswer bankAnswer = BeanUtil.toBean(option, DbQuestionBankAnswer.class);
                    bankAnswer.setQuestionBankId(insertRecord.getId());
                    options.add(bankAnswer);
                }
                answerInsertFlag = questionBankAnswerMapper.insertBatch(options);
            }
        }
        return questionInsertFlag && answerInsertFlag;
    }

    /**
     * 修改题库主
     */
    @Override
    public Boolean updateByBo(DbQuestionBankBo bo) {
        DbQuestionBank update = BeanUtil.toBean(bo, DbQuestionBank.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 修改题库
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean updateByBo(QuestionBankAndAnswerBo bo) {
        DbQuestionBank updateRecord = BeanUtil.toBean(bo, DbQuestionBank.class);
        validEntityBeforeSave(updateRecord);
        boolean questionUpdateFlag = baseMapper.updateById(updateRecord) > 0;
        // 若为选择题则同步插入至选项答案中
        boolean answerUpdateFlag = true;
        if (questionUpdateFlag) {
            // 将新增的题目添加到redis题库缓存中
            List<DbQuestionBank> questionBanks = JsonUtils.parseArray(RedisUtils.getCacheObject(DEFAULT_QUESTIOBN_BANK_KEY), DbQuestionBank.class);
            questionBanks.removeIf(dbQuestionBank -> dbQuestionBank.getId().equals(updateRecord.getId()));
            questionBanks.add(updateRecord);
            RedisUtils.setCacheObject(DEFAULT_QUESTIOBN_BANK_KEY, JsonUtils.toJsonString(questionBanks));

            if (updateRecord.getSpecies() == 2 && CollUtil.isNotEmpty(bo.getOptions())) {
                // 删除选择题旧有数据
                LambdaQueryWrapper<DbQuestionBankAnswer> wrapper = new LambdaQueryWrapper<>();
                Objects.requireNonNull(bo.getId(), "题目ID不能为空");
                wrapper.eq(DbQuestionBankAnswer::getQuestionBankId, bo.getId());
                questionBankAnswerMapper.delete(wrapper);
                //  插入新值
                List<DbQuestionBankAnswer> options = new ArrayList<>(bo.getOptions().size());
                for (DbQuestionBankAnswerBo option : bo.getOptions()) {
                    DbQuestionBankAnswer bankAnswer = BeanUtil.toBean(option, DbQuestionBankAnswer.class);
                    bankAnswer.setQuestionBankId(updateRecord.getId());
                    options.add(bankAnswer);
                }
                answerUpdateFlag = questionBankAnswerMapper.insertBatch(options);
            }
        }
        return questionUpdateFlag && answerUpdateFlag;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(DbQuestionBank entity) {
        // TODO 做一些数据校验,如唯一约束
    }

    /**
     * 批量删除题库主
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if (isValid) {
            // TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteBatchIds(ids) > 0;
    }

    /**
     * 初始化题库到缓存
     */
    @Override
    public void init() {
        List<DbQuestionBank> dbQuestionBanks = baseMapper.selectList(Wrappers.<DbQuestionBank>lambdaQuery().eq(DbQuestionBank::getStatus, 0));
        RedisUtils.setCacheObject("question_bank:default_question_bank", JsonUtils.toJsonString(dbQuestionBanks));
    }
}
