package com.ruoyi.school.paper.controller;

import cn.dev33.satoken.annotation.SaIgnore;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.annotation.RepeatSubmit;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.PageQuery;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.core.validate.AddGroup;
import com.ruoyi.common.core.validate.EditGroup;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.school.paper.domain.bo.DbTestPaperBo;
import com.ruoyi.school.paper.domain.vo.DbTestPaperVo;
import com.ruoyi.school.paper.service.IDbTestPaperService;
import lombok.RequiredArgsConstructor;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Arrays;
import java.util.List;

/**
 * 试卷
 *
 * @author zyf
 * @date 2023-07-20
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/school-paper/testPaper")
@SaIgnore
public class DbTestPaperController extends BaseController {

    private final IDbTestPaperService iDbTestPaperService;

    /**
     * 【小程序】查询试卷列表
     */
    @GetMapping("/list")
    public TableDataInfo<DbTestPaperVo> list(DbTestPaperBo bo, PageQuery pageQuery) {
        return iDbTestPaperService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出试卷列表
     */
    @Log(title = "试卷", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(DbTestPaperBo bo, HttpServletResponse response) {
        List<DbTestPaperVo> list = iDbTestPaperService.queryList(bo);
        ExcelUtil.exportExcel(list, "试卷", DbTestPaperVo.class, response);
    }

    /**
     * 获取试卷详细信息
     *
     * @param id 主键
     */
    @GetMapping("/{id}")
    public R<DbTestPaperVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(iDbTestPaperService.queryById(id));
    }

    /**
     * 新增试卷
     */
    @Log(title = "试卷", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody DbTestPaperBo bo) {
        return toAjax(iDbTestPaperService.insertByBo(bo));
    }

    /**
     * 【PC】随机生成试卷
     */
    @Log(title = "试卷", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("randomGeneration")
    public R<Void> randomGeneration(@Validated(AddGroup.class) @RequestBody DbTestPaperBo bo) {
        return toAjax(iDbTestPaperService.randomGeneration(bo));
    }

    /**
     * 修改试卷
     */
    @Log(title = "试卷", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody DbTestPaperBo bo) {
        return toAjax(iDbTestPaperService.updateByBo(bo));
    }

    /**
     * 删除试卷
     *
     * @param ids 主键串
     */
    @Log(title = "试卷", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(iDbTestPaperService.deleteWithValidByIds(Arrays.asList(ids), true));
    }
}
