package com.ruoyi.school.paper.controller;

import cn.dev33.satoken.annotation.SaIgnore;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.annotation.RepeatSubmit;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.PageQuery;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.core.validate.AddGroup;
import com.ruoyi.common.core.validate.EditGroup;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.school.paper.domain.bo.DbQuestionBankBo;
import com.ruoyi.school.paper.domain.bo.QuestionBankAndAnswerBo;
import com.ruoyi.school.paper.domain.vo.DbQuestionBankAndAnswerVo;
import com.ruoyi.school.paper.domain.vo.DbQuestionBankVo;
import com.ruoyi.school.paper.service.IDbQuestionBankService;
import lombok.RequiredArgsConstructor;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Arrays;
import java.util.List;

/**
 * 题库主
 *
 * @author zyf
 * @date 2023-07-20
 */
@Validated
@RequiredArgsConstructor
@RestController
@SaIgnore
@RequestMapping("/school-paper/questionBank")
public class DbQuestionBankController extends BaseController {

    private final IDbQuestionBankService iDbQuestionBankService;

    /**
     * [PC端]查询题库主列表
     */
    @GetMapping("/list")
    public TableDataInfo<DbQuestionBankVo> list(DbQuestionBankBo bo, PageQuery pageQuery) {
        return iDbQuestionBankService.queryPageList(bo, pageQuery);
    }

    /**
     * [PC端]导出题库主列表
     */
    @Log(title = "题库主", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(DbQuestionBankBo bo, HttpServletResponse response) {
        List<DbQuestionBankVo> list = iDbQuestionBankService.queryList(bo);
        ExcelUtil.exportExcel(list, "题库主", DbQuestionBankVo.class, response);
    }

    /**
     * [PC端]获取题库主详细信息
     *
     * @param id 主键
     */
    @GetMapping("/{id}")
    public R<DbQuestionBankVo> getInfo(@NotNull(message = "主键不能为空") @PathVariable Long id) {
        return R.ok(iDbQuestionBankService.queryById(id));
    }

    /**
     * [PC端]获取题目详细信息
     *
     * @param id 题目ID
     * @return 题目详细信息
     */
    @GetMapping("detail/{id}")
    public R<DbQuestionBankAndAnswerVo> getDetailInfoById(@NotNull(message = "主键不能为空") @PathVariable Long id) {
        return R.ok(iDbQuestionBankService.getById(id));
    }

    /**
     * 新增题库主
     */
    @Log(title = "题库主", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody DbQuestionBankBo bo) {
        return toAjax(iDbQuestionBankService.insertByBo(bo));
    }

    /**
     * [PC端] 新增题库记录
     *
     * @param bo 新增题目信息及选择项信息列表
     * @return 操作结果
     */
    @Log(title = "题库", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/question")
    public R<Void> addQuestionAndAnswerRecords(@Validated(AddGroup.class) @RequestBody QuestionBankAndAnswerBo bo) {
        return toAjax(iDbQuestionBankService.insertByBo(bo));
    }

    /**
     * 修改题库主
     */
    @Log(title = "题库主", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody DbQuestionBankBo bo) {
        return toAjax(iDbQuestionBankService.updateByBo(bo));
    }

    /**
     * [PC端] 修改题目信息
     *
     * @param bo 修改信息
     * @return 操作结果
     */
    @Log(title = "题库", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping("/question")
    public R<Void> editQuestionAndAnswerRecords(@Validated(EditGroup.class) @RequestBody QuestionBankAndAnswerBo bo) {
        return toAjax(iDbQuestionBankService.updateByBo(bo));
    }

    /**
     * [PC端]删除题库主
     *
     * @param ids 主键串
     */
    @Log(title = "题库主", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空") @PathVariable Long[] ids) {
        return toAjax(iDbQuestionBankService.deleteWithValidByIds(Arrays.asList(ids), true));
    }
}
