package com.ruoyi.school.paper.controller;

import java.util.List;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

import cn.dev33.satoken.annotation.SaIgnore;
import lombok.RequiredArgsConstructor;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import com.ruoyi.common.annotation.RepeatSubmit;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.PageQuery;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.validate.AddGroup;
import com.ruoyi.common.core.validate.EditGroup;
import com.ruoyi.common.core.validate.QueryGroup;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.school.paper.domain.vo.DbProfileVo;
import com.ruoyi.school.paper.domain.bo.DbProfileBo;
import com.ruoyi.school.paper.service.IDbProfileService;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 资料
 *
 * @author zyf
 * @date 2023-07-20
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/school-paper/profile")
@SaIgnore
public class DbProfileController extends BaseController {

    private final IDbProfileService iDbProfileService;

    /**
     * 查询资料列表
     */
    @GetMapping("/list")
    public TableDataInfo<DbProfileVo> list(DbProfileBo bo, PageQuery pageQuery) {
        return iDbProfileService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出资料列表
     */
    @Log(title = "资料", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(DbProfileBo bo, HttpServletResponse response) {
        List<DbProfileVo> list = iDbProfileService.queryList(bo);
        ExcelUtil.exportExcel(list, "资料", DbProfileVo.class, response);
    }

    /**
     * 获取资料详细信息
     *
     * @param id 主键
     */
    @GetMapping("/{id}")
    public R<DbProfileVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(iDbProfileService.queryById(id));
    }

    /**
     * 新增资料
     */
    @Log(title = "资料", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody DbProfileBo bo) {
        return toAjax(iDbProfileService.insertByBo(bo));
    }

    /**
     * 修改资料
     */
    @Log(title = "资料", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody DbProfileBo bo) {
        return toAjax(iDbProfileService.updateByBo(bo));
    }

    /**
     * 删除资料
     *
     * @param ids 主键串
     */
    @Log(title = "资料", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(iDbProfileService.deleteWithValidByIds(Arrays.asList(ids), true));
    }
}
