package com.ruoyi.school.paper.controller;

import java.util.List;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

import cn.dev33.satoken.annotation.SaIgnore;
import com.ruoyi.school.paper.domain.request.ErrorTopicRequest;
import com.ruoyi.school.paper.domain.vo.ErrorTopicListDetailVo;
import com.ruoyi.school.paper.domain.vo.ErrorTopicListVo;
import com.ruoyi.school.paper.domain.vo.MyErrorTopicListVo;
import lombok.RequiredArgsConstructor;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.*;

import cn.dev33.satoken.annotation.SaCheckPermission;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import com.ruoyi.common.annotation.RepeatSubmit;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.PageQuery;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.validate.AddGroup;
import com.ruoyi.common.core.validate.EditGroup;
import com.ruoyi.common.core.validate.QueryGroup;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.school.paper.domain.vo.DbErrorListVo;
import com.ruoyi.school.paper.domain.bo.DbErrorListBo;
import com.ruoyi.school.paper.service.IDbErrorListService;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 错题集
 *
 * @author zyf
 * @date 2023-07-20
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/school-paper/errorList")
@SaIgnore
public class DbErrorListController extends BaseController {

    private final IDbErrorListService iDbErrorListService;

    /**
     * 【后台】查询错题集列表
     */
    @GetMapping("/list")
    public TableDataInfo<ErrorTopicListVo> findErrorTopicPage(ErrorTopicRequest request, PageQuery pageQuery) {
        return iDbErrorListService.findErrorTopicPage(pageQuery, request);
    }

    /**
     * 【小程序】我的错题集列表
     */
    @GetMapping("/MyErrorTopicList")
    public TableDataInfo<MyErrorTopicListVo> findMyErrorTopicListVoPage(ErrorTopicRequest request, PageQuery pageQuery) {
        request.setUserId(1L);
        return iDbErrorListService.findMyErrorTopicListVoPage(pageQuery, request);
    }

    /**
     * 导出错题集列表
     */
    @Log(title = "错题集", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(DbErrorListBo bo, HttpServletResponse response) {
        List<DbErrorListVo> list = iDbErrorListService.queryList(bo);
        ExcelUtil.exportExcel(list, "错题集", DbErrorListVo.class, response);
    }

    /**
     * 获取错题集详细信息
     *
     * @param id 主键
     */
    @GetMapping("/{id}")
    public R<ErrorTopicListDetailVo> getInfo(@NotNull(message = "主键不能为空")
                                             @PathVariable Long id) {
        return R.ok(iDbErrorListService.queryById(id));
    }

    /**
     * 新增错题集
     */
    @Log(title = "错题集", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody DbErrorListBo bo) {
        return toAjax(iDbErrorListService.insertByBo(bo));
    }

    /**
     * 修改错题集
     */
    @Log(title = "错题集", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody DbErrorListBo bo) {
        return toAjax(iDbErrorListService.updateByBo(bo));
    }

    /**
     * 删除错题集
     *
     * @param ids 主键串
     */
    @Log(title = "错题集", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(iDbErrorListService.deleteWithValidByIds(Arrays.asList(ids), true));
    }
}
