package com.ruoyi.school.paper.controller;

import java.util.List;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

import cn.dev33.satoken.annotation.SaIgnore;
import com.ruoyi.school.paper.domain.vo.MyCollectListVo;
import lombok.RequiredArgsConstructor;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.*;

import cn.dev33.satoken.annotation.SaCheckPermission;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import com.ruoyi.common.annotation.RepeatSubmit;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.PageQuery;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.validate.AddGroup;
import com.ruoyi.common.core.validate.EditGroup;
import com.ruoyi.common.core.validate.QueryGroup;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.school.paper.domain.vo.DbCollectListVo;
import com.ruoyi.school.paper.domain.bo.DbCollectListBo;
import com.ruoyi.school.paper.service.IDbCollectListService;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 收藏
 *
 * @author zyf
 * @date 2023-07-20
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/school-paper/collectList")
@SaIgnore
public class DbCollectListController extends BaseController {

    private final IDbCollectListService iDbCollectListService;

    /**
     * 查询我的收藏列表
     */
    @GetMapping("/list")
    public TableDataInfo<MyCollectListVo> list(@RequestParam("categoryId") Integer categoryId, PageQuery pageQuery) {
        return iDbCollectListService.queryPageList(categoryId, getUserId(), pageQuery);
    }

    /**
     * 导出收藏列表
     */
    @Log(title = "收藏", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(DbCollectListBo bo, HttpServletResponse response) {
        List<DbCollectListVo> list = iDbCollectListService.queryList(bo);
        ExcelUtil.exportExcel(list, "收藏", DbCollectListVo.class, response);
    }

    /**
     * 获取收藏详细信息
     *
     * @param id 主键
     */
    @GetMapping("/{id}")
    public R<DbCollectListVo> getInfo(@NotNull(message = "主键不能为空")
                                      @PathVariable Long id) {
        return R.ok(iDbCollectListService.queryById(id));
    }

    /**
     * 【小程序】新增收藏
     *
     * @param bo 参数
     * @return
     */
    @Log(title = "收藏", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody DbCollectListBo bo) {
        return toAjax(iDbCollectListService.insertByBo(bo));
    }

    /**
     * 修改收藏
     */
    @Log(title = "收藏", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody DbCollectListBo bo) {
        return toAjax(iDbCollectListService.updateByBo(bo));
    }

    /**
     * 删除收藏
     *
     * @param ids 主键串
     */
    @Log(title = "收藏", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(iDbCollectListService.deleteWithValidByIds(Arrays.asList(ids), true));
    }
}
