<?php

namespace App\Store\Controllers;

use App\Models\StoreGood;
use App\Models\StoreGoodsCategory;
use App\Models\StoreGoodsTag;
use Dcat\Admin\Admin;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;

class StoreGoodController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(StoreGood::with('store_category'), function (Grid $grid) {
            $grid->addTableClass(['table-text-center']);
            $grid->model()->where(['store_id'=>Admin::user()->id])->orderBy('created_at','DESC');
            $grid->column('id')->sortable();
            $grid->column('title');
            $grid->column('store_category.title','商品类别')->label('default');
            $grid->column('tags')->display(function ($vel) {
                if ($vel) {
                    $html = '';
                    $rel = json_decode($vel,true);
                    foreach ($rel as $k=>$t) {
                        $html .= '<span class="badge" style="background:green">' . StoreGoodsTag::find($t)->title . '</span>';
                        if((($k+1)%2) == 0){
                            $html .= '<br />';
                        }
                    }
                    return $html;
                } else {
                    return '未设置';
                }
            });

            $grid->column('price');
            $grid->column('sale_price');
            $grid->column('small_cover')->image(env('IMAGE_URL'),50);
            $grid->column('cover')->image(env('IMAGE_URL'),50);

            $grid->column('intro')->if(function ($column){
                return $column->getValue();
            })->then(function (Grid\Column $column){
                $column->display('点击查看')->modal(function (Grid\Displayers\Modal $modal){
                    $modal->title('商品描述');
                    $modal->xl();
                    return $this->intro;
                });
            })->else(function (Grid\Column $column) {
                $column->display('未设置');
            });
            $grid->column('look_num');
            $grid->column('inventory');
            $grid->column('updated_at')->sortable();


            $grid->simplePaginate();

            $grid->setDialogFormDimensions('70%', '90%');
            $grid->enableDialogCreate();
            $grid->showQuickEditButton();
            $grid->disableEditButton();
            $grid->disableViewButton();
            $grid->simplePaginate();
            $grid->disableFilterButton();
            $grid->quickSearch('title')->placeholder('搜索商品');
            $grid->filter(function (Grid\Filter $filter) {
                $filter->equal('id');

            });
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new StoreGood(), function (Show $show) {
            $show->field('id');
            $show->field('title');
            $show->field('tags');
            $show->field('price');
            $show->field('sale_price');
            $show->field('small_cover');
            $show->field('cover');
            $show->field('category_id');
            $show->field('intro');
            $show->field('look_num');
            $show->field('inventory');
            $show->field('created_at');
            $show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new StoreGood(), function (Form $form) {
            if($form->isCreating()){
                $form->hidden('store_id')->value(Admin::user()->id);
            }
            $form->text('title');
            $form->multipleSelect('tags')
                ->options(StoreGoodsTag::where(['store_id'=>Admin::user()->id])->get()->pluck('title','id'))
                ->saving(function ($value) {
                    return $value ? json_encode($value) : null;
                });
            $form->currency('price')->symbol('¥');
            $form->currency('sale_price')->symbol('¥');
            $form->image('small_cover')->accept('jpg,jpeg,png')
                ->maxSize(4096)
                ->url('upload/store')
                ->help('仅支持jpg、jpeg、png格式图片上传')
                ->autoUpload();
            $form->image('cover')->accept('jpg,jpeg,png')
                ->maxSize(4096)
                ->url('upload/store')
                ->help('仅支持jpg、jpeg、png格式图片上传')
                ->autoUpload();
            $form->select('category_id')
                ->options(StoreGoodsCategory::where(['store_id'=>Admin::user()->id])->get()->pluck('title','id'))
                ->default(0)->required();
            $form->editor('intro') ->imageUrl('upload/store')
                ->height(400);

            $form->number('inventory');

            $form->disableDeleteButton();
            $form->disableEditingCheck();
            $form->disableViewButton();
            $form->disableViewCheck();
            $form->disableCreatingCheck();
        });
    }
}
