<?php

namespace App\Admin\Controllers;

use App\Admin\Repositories\Lawyer;
use App\Models\Lawyer as ModelLawyer;
use App\Models\LawyerCost as ModelLawyerCost;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use Illuminate\Support\Facades\DB;
use App\Models\CovenantReceivePayment as ModelCovenantReceivePayment;

class LawyerCommissionController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new Lawyer(), function (Grid $grid) {
            $year = $grid->model()->filter()->input('year') ?? date('Y');
            //$grid->column('id')->sortable();
            $grid->column('number', '律师编号');
            $grid->column('name', '律师姓名');
            $grid->column('received_money', '创收已收款')->display(function ($val) use ($year) {
                $money = self::getReceivedMoney($year, $this->id);
                return $money;
            });
            $grid->column('commission_rate', '提成比例 %');
            $grid->column('paid_amount', '已支付款项')->display(function ($val) use ($year) {
                $paid_amount =  self::getPaidAmount($year, $this->id);
                return sprintf('%.2f', $paid_amount);
            });
            $grid->column('posting_tickets_money', '贴票金额')->display(function () use ($year) {
                $tickets_money = ModelLawyerCost::getPostingTicketsMoney($year, $this->id, $this->commission_rate, $this->ticket_ratio);
                return $tickets_money;
            });
            $grid->column('commission_retention', '提成留底')->display(function () use ($year) {
                $received_money = ModelCovenantReceivePayment::getReceivedMoney($this->id, $year); //创收已收款
                $commission_rate = $this->commission_rate; //提成比例
                $result = $received_money * ($commission_rate / 100) * 0.05;
                return sprintf('%.2f', $result);
            });
            $grid->column('payable_amount', '可支付提成结算金额')->display(function () use ($year) {
                $received_money = ModelCovenantReceivePayment::getReceivedMoney($this->id, $year); //创收已收款
                $paid_amount =  ModelLawyerCost::getPaidAmount($this->id, $year); //已支付款项
                $commission_rate = $this->commission_rate; //提成比例
                $result = $received_money * ($commission_rate / 100) - $paid_amount;
                return sprintf('%.2f', $result);
            });
            $grid->column('personal_income_tax', '个人所得税')->display(function ($val) use ($year) {
                $amount =  self::getPersonalIncomeTax($year, $this->id);
                return sprintf('%.2f', $amount);
            });

            // $grid->column('created_at');
            // $grid->column('updated_at')->sortable();
            $grid->disableCreateButton();
            $grid->disableViewButton();
            $grid->disableEditButton();
            $grid->disableDeleteButton();
            $grid->disableRowSelector();

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->like('number', '律师编号')->width(3);
                $filter->like('name', '律师姓名')->width(3);
                $filter->like('year', '年份')->width(3)->default(date("Y"))->ignore();
            });

            $grid->actions(function (Grid\Displayers\Actions $actions) {
                //查看
                $actions->append('<a href="/lawyer-commission-collect?no=' . $this->number . '" alt="查看" target="_blank">查看</a>');
            });
        });
    }
    //创收已收款
    public static function getReceivedMoney($year, $lawyer_id)
    {
        $money = ModelCovenantReceivePayment::where(['lawyer_id' => $lawyer_id, 'rtype' => 1])
            ->where('received_at', 'like', $year . '%')
            ->sum('received_amount');
        return $money;
    }
    //已支付款项
    public static function getPaidAmount($year, $lawyer_id)
    {
        $paid_amount = 0;
        $list = ModelLawyerCost::where(['lawyer_id' => $lawyer_id, 'year' => $year])->get();
        if ($list->toArray()) {
            $basic_salary = $special_additional = $social_company_fee = $accumulation_fund_company_fee = 0;
            $annual_inspection_fee = $annuity = $office_rental_fee = $assistant_fee = 0;
            foreach ($list as $item) {
                $basic_salary += $item->basic_salary;
                $special_additional += $item->special_additional;
                $social_company_fee += $item->social_company_fee;
                $accumulation_fund_company_fee += $item->accumulation_fund_company_fee;
                $annual_inspection_fee += $item->annual_inspection_fee;
                $annuity += $item->annuity;
                $office_rental_fee += $item->office_rental_fee;
                $assistant_fee += $item->assistant_fee;
            }
            $paid_amount = $basic_salary + $special_additional + $social_company_fee + $accumulation_fund_company_fee + $annual_inspection_fee + $annuity + $office_rental_fee + $assistant_fee;
        }
        return $paid_amount;
    }
    //个人所得税
    public static function getPersonalIncomeTax($year, $lawyer_id)
    {
        $amount = 0;
        $list = ModelLawyerCost::where(['lawyer_id' => $lawyer_id, 'year' => $year])->get();
        if ($list->toArray()) {
            $personal_income_tax = 0;
            foreach ($list as $item) {
                $amount += $item->personal_income_tax;
            }
        }
        return $amount;
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new Lawyer(), function (Show $show) {
            $show->field('id');
            $show->field('number');
            $show->field('name');
            $show->field('sex');
            $show->field('identity_card');
            $show->field('phone');
            $show->field('bank_account');
            $show->field('sign_at');
            $show->field('commission_rate');
            $show->field('settlement_period');
            $show->field('created_at');
            $show->field('updated_at');
        });
    }
}
