<?php

namespace App\Admin\Controllers;

use App\Admin\Repositories\Covenant;
use App\Models\Covenant as ModelsCovenant;
use App\Models\Lawyer as ModelLawyer;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use App\Admin\Actions\CovenantClosed;
use App\Admin\Actions\CovenantReturn;
use Dcat\Admin\Admin;

class CovenantController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(ModelsCovenant::with(['principal', 'lawyer']), function (Grid $grid) {
            $grid->model()->orderBy('id', 'DESC');
            $grid->column('id')->sortable();
            $grid->column('number');
            $grid->column('cname');
            $grid->column('ctype')->display(function ($val) {
                return Covenant::CTYPE[$val];
            });
            $grid->column('sign_at');
            $grid->column('principal.name', '委托人');
            $grid->column('lawyer.name', '办案律师');
            $grid->column('amount');
            $grid->column('payment_method')->display(function ($val) {
                return $val == 1 ? '开票付款' : '付款开票';
            });
            $grid->column('reserve_fee', '预留结案费');
            $grid->column('is_closed', '结案状态')->display(function ($val) {
                return $val ? '已结案' : '未结案';
            });
            $grid->column('is_return', '退还状态')->display(function ($val) {
                return $val ? '已退还' : '未退还';
            });
            $grid->column('return_fee', '退还结案费');
            // $grid->column('created_at');
            // $grid->column('updated_at')->sortable();

            $grid->disableViewButton();
            $grid->disableRowSelector();

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->like('number')->width(3);
                $filter->like('cname')->width(3);
                $filter->like('principal.name', '委托人')->width(3);
                $filter->like('lawyer.name', '办案律师')->width(3);
                $filter->equal('ctype')->select(Covenant::CTYPE)->width(3);
                $filter->equal('payment_method')->select(Covenant::PAYMENT_METHOD)->width(3);
                $filter->between('sign_at', '签约日期')->date()->width(4);
            });

            $grid->actions(function (Grid\Displayers\Actions $actions) {
                //编辑
                if (!Admin::user()->can('hetong-edit')) {
                    $actions->disableEdit();
                }
                //删除
                if (!Admin::user()->can('hetong-delete')) {
                    $actions->disableDelete();
                }
                //已结案
                if (Admin::user()->can('hetong-closed')) {
                    $actions->append(new CovenantClosed($actions->row->id));
                }
                // 已退还
                if (Admin::user()->can('hetong-return')) {
                    $actions->append(new CovenantReturn($actions->row->id));
                }
            });
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new Covenant(), function (Show $show) {
            $show->field('id');
            $show->field('number');
            $show->field('cname');
            $show->field('ctype');
            $show->field('sign_at');
            $show->field('principal_id');
            $show->field('lawyer_id');
            $show->field('amount');
            $show->field('payment_method');
            $show->field('case_reason');
            $show->field('created_at');
            $show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {

        $list = ModelLawyer::select(['id', 'number', 'name'])->limit(1000)->get()->toArray();
        $lawyers = [];
        if ($list) {
            foreach ($list as $item) {
                $lawyers[$item['id']] = $item['number'] . ' ' . $item['name'];
            }
        }

        return Form::make(new Covenant(), function (Form $form) use ($lawyers) {
            $form->display('id');
            $form->text('number')->required();
            $form->select('ctype')->options(Covenant::CTYPE)->default(1);
            $form->text('cname')->required();
            $form->date('sign_at')->format('YYYY-MM-DD')->default(date("Y-m-d"));
            $form->select('principal_type')->options([1 => '单位', 2 => '个人'])->default(1)->load('principal_id', '/principal-list');;
            $form->select('principal_id');
            $form->text('case_reason');
            $form->select('lawyer_id')->options($lawyers);
            $form->text('amount');
            $form->select('payment_method')->options(Covenant::PAYMENT_METHOD)->default(1);

            $form->disableCreatingCheck();
            $form->disableEditingCheck();
            $form->disableViewCheck();
            $form->disableDeleteButton();
            $form->disableViewButton();

            $form->display('created_at');
            $form->display('updated_at');
        });
    }
}
