<?php

namespace App\Admin\Controllers;

use App\Admin\Repositories\Covenant;
use App\Models\Covenant as ModelsCovenant;
use App\Models\Lawyer as ModelLawyer;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use App\Admin\Actions\CovenantReceive;
use App\Admin\Actions\CovenantInvoice;
use Dcat\Admin\Admin;

class LawyerCovenantIncomeController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new Covenant(), function (Grid $grid) {
            $lawyerID = $grid->model()->filter()->input('no') ?? ''; //律师ID
            if ($lawyerID) {
                $grid->model()->where('lawyer_id', $lawyerID);
            }
            //$grid->column('cid')->sortable();
            $grid->column('number', '合同编号');
            $grid->column('cname', '合同名称');
            $grid->column('ctype', '合同类型')->display(function ($val) {
                return ModelsCovenant::CTYPE[$val];
            });
            $grid->column('principal', '委托人');
            $grid->column('invoice_amount', '开票金额');
            $grid->column('receipt_money', '已收款');
            $grid->column('receipt_noinvoice', '已收款未开票');
            $grid->column('invoice_noreceipt', '已开票未收款');

            // $grid->column('created_at');
            // $grid->column('updated_at')->sortable();

            $grid->disableViewButton();
            $grid->disableCreateButton();
            $grid->disableEditButton();
            $grid->disableDeleteButton();
            $grid->disableRowSelector();

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->like('number', '合同编号')->width(3);
                $filter->like('cname', '合同名称')->width(3);
                $filter->like('principal', '委托人')->width(3);
                $filter->between('received_at', '收款日期')->date()->width(4);
                $filter->between('invoiced_at', '开票日期')->date()->width(4);
            });

            //文字信息
            $grid->tools(function (Grid\Tools $tools) use ($grid, $lawyerID) {
                $fullname = $number = '';
                if ($lawyerID) {
                    $obj = ModelLawyer::find($lawyerID);
                    $fullname = $obj->name;
                    $number = $obj->number;
                }
                $card_info = "&nbsp;&nbsp;律师编号：{$number}&nbsp;&nbsp;&nbsp;律师姓名：{$fullname}";

                $tools->append($card_info);
            });
            $grid->actions(function (Grid\Displayers\Actions $actions) use ($lawyerID) {
                //添加收款
                if (Admin::user()->can('receive-payment')) {
                    $actions->append(new CovenantReceive('添加收款', $actions->row->cid));
                }
                //添加开票
                if (Admin::user()->can('invoiced-payment')) {
                    $actions->append(new CovenantInvoice('添加开票', $actions->row->cid));
                }
                //查看收款
                $cid = $actions->row->cid;
                if (Admin::user()->can('view-received')) {
                    $actions->append('<a href="/view-covenant-received?cid=' . $cid . '" alt="查看收款" target="_blank">查看收款</a>');
                }
                //查看开票
                if (Admin::user()->can('view-invoiced')) {
                    $actions->append('<a href="/view-covenant-invoiced?cid=' . $cid . '" alt="查看开票" target="_blank">查看开票</a>');
                }
            });
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new Covenant(), function (Show $show) {
            $show->field('id');
            $show->field('number');
            $show->field('cname');
            $show->field('ctype');
            $show->field('sign_at');
            $show->field('principal_id');
            $show->field('lawyer_id');
            $show->field('amount');
            $show->field('payment_method');
            $show->field('case_reason');
            $show->field('created_at');
            $show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {

        $list = ModelLawyer::select(['id', 'number', 'name'])->limit(1000)->get()->toArray();
        $lawyers = [];
        if ($list) {
            foreach ($list as $item) {
                $lawyers[$item['id']] = $item['number'] . ' ' . $item['name'];
            }
        }

        return Form::make(new Covenant(), function (Form $form) use ($lawyers) {
            $form->display('id');
            $form->text('number')->required();
            $form->select('ctype')->options(Covenant::CTYPE)->default(1);
            $form->text('cname')->required();
            $form->date('sign_at')->format('YYYY-MM-DD')->default(date("Y-m-d"));
            $form->select('principal_type')->options([1 => '单位', 2 => '个人'])->default(1)->load('principal_id', '/principal-list');;
            $form->select('principal_id');
            $form->text('case_reason');
            $form->select('lawyer_id')->options($lawyers);
            $form->text('amount');
            $form->select('payment_method')->options(Covenant::PAYMENT_METHOD)->default(1);

            $form->disableCreatingCheck();
            $form->disableEditingCheck();
            $form->disableViewCheck();
            $form->disableDeleteButton();
            $form->disableViewButton();

            $form->display('created_at');
            $form->display('updated_at');
        });
    }
}
