<?php

namespace App\Admin\Controllers;

//use App\Admin\Repositories\LawyerCost;
use App\Admin\Repositories\Lawyer;
use App\Models\LawyerCost as ModelLawyerCost;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use Illuminate\Support\Facades\DB;
use Dcat\Admin\Admin;

class LawyerCostController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new Lawyer(), function (Grid $grid) {
            $number = $grid->model()->filter()->input('number') ?? '';
            $name = $grid->model()->filter()->input('name') ?? '';
            $year = $_GET['year'] ?? date("Y");
            //$grid->column('id')->sortable();
            $grid->column('number', '律师编号');
            $grid->column('name', '律师姓名');
            $grid->column('basic_salary', '基本工资')->display(function ($val) use ($year) {
                return self::getBasiSalary($this->number, $year);
            });
            $grid->column('special_additional', '专项附加')->display(function ($val) use ($year) {
                return self::getSpecialAdditional($this->number, $year);
            });
            $grid->column('social', '社保')->display(function ($val) use ($year) {
                return self::getSocial($this->number, $year);
            });
            $grid->column('accumulation_fund', '公积金')->display(function ($val) use ($year) {
                return self::getAccumulationFund($this->number, $year);
            });
            $grid->column('annual_inspection_fee', '律所年检费')->display(function ($val) use ($year) {
                return self::getAnnualInspectionFee($this->number, $year);
            });
            $grid->column('annuity', '律所年金')->display(function ($val) use ($year) {
                return self::getAnnuity($this->number, $year);
            });
            $grid->column('posting_tickets_fee', '贴票成本')->display(function ($val) use ($year) {
                return self::getAssistantFee($this->number, $year);
            });
            $grid->column('assistant_fee', '助理律师成本')->display(function ($val) use ($year) {
                return self::getAssistantFee($this->number, $year);
            });
            $grid->column('office_rental_fee', '办公租赁成本')->display(function ($val) use ($year) {
                return self::getOfficeRentalFee($this->number, $year);
            });
            $grid->column('noticket_cost', '无票成本')->display(function ($val) use ($year) {
                return self::getNoticketCost($this->number, $year);
            });
            $grid->column('personal_income_tax', '个人所得税')->display(function ($val) use ($year) {
                return self::getPersonalIncomeTax($this->number, $year);
            });

            if (!Admin::user()->can('lawyer_cost_add')) {
                $grid->disableCreateButton();
            }
            $grid->disableViewButton();
            $grid->disableEditButton();
            $grid->disableDeleteButton();
            $grid->disableRowSelector();

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->like('number', '律师编号')->width(3);
                $filter->like('name', '律师姓名')->width(3);
                $filter->like('year', '年份')->width(3)->default(date("Y"))->ignore();
                //$filter->date('sign_at')->format('YYYY-MM-DD');
            });

            $grid->actions(function (Grid\Displayers\Actions $actions) {
                //成本汇总
                if (Admin::user()->can('lycost-collect')) {
                    $actions->append('<a href="/lycost-collect?no=' . $this->number . '" alt="成本汇总" target="_blank">成本汇总</a>');
                }

                // 成本明细
                if (Admin::user()->can('lycost-list')) {
                    $actions->append('<a href="/lycost-list?no=' . $this->number . '" alt="成本明细" target="_blank">成本明细</a>');
                }
            });
        });
    }

    //基本工资
    public static function getBasiSalary($number, $year)
    {
        return ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('basic_salary');
    }
    //专项附加 
    public static function  getSpecialAdditional($number, $year)
    {
        return ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('special_additional');
    }

    //社保
    public static function  getSocial($number, $year)
    {
        $social_person_fee =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('social_person_fee');
        $social_company_fee =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('social_company_fee');
        return round($social_person_fee + $social_company_fee, 2);
    }


    //公积金 
    public static function  getAccumulationFund($number, $year)
    {
        $accumulation_fund_person_fee =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('accumulation_fund_person_fee');
        $accumulation_fund_company_fee =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('accumulation_fund_company_fee');
        return round($accumulation_fund_person_fee + $accumulation_fund_company_fee, 2);
    }

    //律所年检费 
    public static function  getAnnualInspectionFee($number, $year)
    {
        $annual_inspection_fee =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('annual_inspection_fee');
        return $annual_inspection_fee;
    }

    //律所年金 
    public static function  getAnnuity($number, $year)
    {
        $annuity =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('annuity');
        return $annuity;
    }

    //助理律师成本  
    public static function  getAssistantFee($number, $year)
    {
        $annuity =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('assistant_fee');
        return $annuity;
    }

    //办公室租赁成本  
    public static function  getOfficeRentalFee($number, $year)
    {
        $office_rental_fee =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('office_rental_fee');
        return $office_rental_fee;
    }

    //个人所得税  
    public static function  getPersonalIncomeTax($number, $year)
    {
        $personal_income_tax =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('personal_income_tax');
        return $personal_income_tax;
    }

    //无票成本  
    public static function  getNoticketCost($number, $year)
    {
        $noticket_cost =  ModelLawyerCost::where(['number' => $number, 'year' => $year])->sum('noticket_cost');
        return $noticket_cost;
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new ModelLawyerCost(), function (Show $show) {
            $show->field('id');
            $show->field('year');
            $show->field('month');
            $show->field('number');
            $show->field('lname');
            $show->field('basic_salary');
            $show->field('social_person_fee');
            $show->field('social_company_fee');
            $show->field('accumulation_fund_person_fee');
            $show->field('accumulation_fund_company_fee');
            $show->field('annual_inspection_fee');
            $show->field('annuity');
            $show->field('office_rental_fee');
            $show->field('noticket_cost');
            $show->field('posting_tickets_fee');
            $show->field('assistant_fee');
            $show->field('special_additional');
            $show->field('advance_fee');
            $show->field('personal_income_tax');
            $show->field('created_at');
            $show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        $form = Form::make(new ModelLawyerCost(), function (Form $form) {
            $form->display('id');
            $form->text('year')->default(date('Y'));
            $form->select('month')->options(ModelLawyerCost::MONTH)->required();
            $form->text('number')->required();
            $form->text('lname')->required();
            $form->text('basic_salary')->required();
            $form->text('special_additional')->required();
            $form->text('social_person_fee')->required();
            $form->text('social_company_fee')->required();
            $form->text('social', '社保')->readOnly();
            $form->text('accumulation_fund_person_fee')->required();
            $form->text('accumulation_fund_company_fee')->required();
            $form->text('accumulation_fund', '公积金')->readOnly();
            $form->text('annual_inspection_fee')->required();
            $form->text('annuity')->required();
            $form->text('office_rental_fee')->required();
            $form->text('noticket_cost')->readOnly();
            //$form->text('posting_tickets_fee');
            $form->text('assistant_fee')->required();
            $form->text('special_additional')->required();
            $form->text('advance_fee');
            $form->text('personal_income_tax')->required();

            $form->disableCreatingCheck();
            $form->disableEditingCheck();
            $form->disableViewCheck();
            $form->disableDeleteButton();
            $form->disableViewButton();

            $form->display('created_at');
            $form->display('updated_at');
        });

        Admin::script(
            <<<JS
            $(function () {
                //社保
                $("input[name='social_company_fee']").on('blur', function() {
                    var person_fee = $("input[name='social_person_fee']").val().length>0?$("input[name='social_person_fee']").val():0;
                    var company_fee = $("input[name='social_company_fee']").val().length>0?$("input[name='social_company_fee']").val():0;
                    var total = parseFloat(person_fee) + parseFloat(company_fee);
                    $("input[name='social']").val(total);
                });
                //公积金
                $("input[name='accumulation_fund_company_fee']").on('blur', function() {
                    var person_fee = $("input[name='accumulation_fund_person_fee']").val().length>0?$("input[name='accumulation_fund_person_fee']").val():0;
                    var company_fee = $("input[name='accumulation_fund_company_fee']").val().length>0?$("input[name='accumulation_fund_company_fee']").val():0;
                    var accumulation_fund = parseFloat(person_fee) + parseFloat(company_fee);
                    $("input[name='accumulation_fund']").val(accumulation_fund);
                });
                //无票成本
                $("input[name='office_rental_fee']").on('blur', function() {
                    //基本工资
                    var salary = $("input[name='basic_salary']").val().length>0?$("input[name='basic_salary']").val():0;
                    //专项附加
                    var additional = $("input[name='special_additional']").val().length>0?$("input[name='special_additional']").val():0;
                    //社保单位部分
                    var social_company = $("input[name='social_company_fee']").val().lenght>0?$("input[name='social_company_fee']").val():0;
                    //公积金单位部分
                    var accumulation_fund = $("input[name='accumulation_fund_company_fee']").val().length>0?$("input[name='accumulation_fund_company_fee']").val():0;
                    //助理律师成本
                    var assistant_fee = $("input[name='assistant_fee']").val().length>0?$("input[name='assistant_fee']").val():0;
                    //办公室租赁成本
                    var office_rental_fee = $("input[name='office_rental_fee']").val().length>0?$("input[name='office_rental_fee']").val():0;

                    var cost = parseFloat(salary) + parseFloat(additional)+ parseFloat(social_company)+ parseFloat(accumulation_fund)+ parseFloat(assistant_fee)+ parseFloat(office_rental_fee);
                    $("input[name='noticket_cost']").val(cost);
                });
            });
            JS
        );

        //副表保存规格
        $form->saved(
            function (Form $form, $result) {
                $cost_id = $form->getKey();

                //律师ID
                $number = $_POST['number'] ?? 0;
                $lawyer_id = DB::table('lawyer')->where('number', $number)->value('id');

                $basic_salary = $_POST['basic_salary'] ?? 0;
                $special_additional = $_POST['special_additional'] ?? 0;
                $social_company_fee = $_POST['social_company_fee'] ?? 0;
                $accumulation_fund_company_fee = $_POST['accumulation_fund_company_fee'] ?? 0;
                $assistant_fee = $_POST['assistant_fee'] ?? 0;
                $office_rental_fee = $_POST['office_rental_fee'] ?? 0;
                //无票成本
                $noticket_cost = floatval($basic_salary) + floatval($special_additional) + floatval($social_company_fee) + floatval($accumulation_fund_company_fee) + floatval($assistant_fee) + floatval($office_rental_fee);
                //社保
                $social_person_fee = $_POST['social_person_fee'] ?? 0;
                $social = floatval($social_person_fee) + floatval($social_company_fee);
                //公积金
                $accumulation_fund_person_fee = $_POST['accumulation_fund_person_fee'] ?? 0;
                $accumulation_fund = floatval($accumulation_fund_person_fee) + floatval($accumulation_fund_company_fee);
                if ($result) {
                    DB::table('lawyer_cost')->where("id", $cost_id)->update(['lawyer_id' => $lawyer_id, 'social' => $social, 'accumulation_fund' => $accumulation_fund, 'noticket_cost' => $noticket_cost]);
                }
            }
        );

        return $form;
    }
}
