<?php

namespace App\Admin\Controllers;

use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use App\Models\UserPointChangeRec;

class UserPointChangeRecController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(UserPointChangeRec::with('user'), function (Grid $grid) {

            $grid->model()->orderBy('created_at', 'DESC');
            $grid->column('id', 'ID')->sortable();
            $grid->column('user_id', '用户ID');
            // 添加用户id字段
            $grid->column('user_id', '用户ID')->display(function ($userId) {
                return "<a href='/user?id={$userId}' >{$userId}</a>";
            });
            // 添加用户昵称字段
            $grid->column('user.name', '用户昵称')->display(function ($name) {
                return "<a href='/user?id={$this->user_id}' >{$name}</a>";
            });

            $grid->column('order_id', '关联订单号');
            $grid->column('point_amount', '积分数量')->display(function ($value) {
                $prefix = $this->change_type == 1 ? '+' : '-';
                $color = $this->change_type == 1 ? 'green' : 'red';
                return "<span style='color:{$color}'>$prefix$value</span>";
            });
            $grid->column('change_type', '增减类别')->using([1 => '增加', 0 => '减少']);
            $grid->column('point_state', '积分状态')->using(UserPointChangeRec::pointState);
            $grid->column('freeze_end_date', '解冻结束日期');
            $grid->column('source', '积分来源')->using(UserPointChangeRec::source);
            $grid->column('memo', '备注');
            $grid->column('created_at', '创建时间')->sortable();

            $grid->filter(function ($filter) {
                $filter->panel();
                $filter->equal('user_id', '用户ID')->width(3);
                // 添加用户昵称筛选
                $filter->like('user.name', '用户昵称')->width(3);
                $filter->equal('change_type', '增减类别')->radio([
                    1 => '增加',
                    0 => '减少'
                ])->width(4);
                $filter->equal('point_state', '积分状态')->select([
                    1 => '已完成',
                    2 => '解冻中',
                    3 => '冻结中'
                ])->width(3);

                // 添加时间范围筛选
                $filter->scope('today', '今日')->whereDate('created_at', date('Y-m-d'));
                $filter->scope('month', '本月')->whereMonth('created_at', date('m'));
                $filter->scope('quarter', '本季度')->where(function ($query) {
                    $month = date('m');
                    $year = date('Y');
                    
                    if ($month >= 1 && $month <= 3) {
                        // 第一季度 (1-3月)
                        $start = date('Y-m-d', strtotime('first day of January'));
                        $end = date('Y-m-d', strtotime('last day of March'));
                    } elseif ($month >= 4 && $month <= 6) {
                        // 第二季度 (4-6月)
                        $start = date('Y-m-d', strtotime('first day of April'));
                        $end = date('Y-m-d', strtotime('last day of June'));
                    } elseif ($month >= 7 && $month <= 9) {
                        // 第三季度 (7-9月)
                        $start = date('Y-m-d', strtotime('first day of July'));
                        $end = date('Y-m-d', strtotime('last day of September'));
                    } else {
                        // 第四季度 (10-12月)
                        $start = date('Y-m-d', strtotime('first day of October'));
                        $end = date('Y-m-d', strtotime('last day of December'));
                    }
                    
                    $query->whereBetween('created_at', [$start, $end]);
                });
                $filter->scope('year', '本年')->whereYear('created_at', date('Y'));

                $filter->between('created_at', '自定义时间范围')->datetime()->width(5);
            });

            // 添加导出按钮
            $grid->export()->filename('积分记录-' . date('YmdHis'))->rows(function ($rows) {
                foreach ($rows as $index => &$row) {
                    // 处理增减类别
                    $row['change_type'] = $row['change_type'] == 1 ? '增加' : '减少';

                    // 处理积分状态
                    $pointStateMap = [
                        1 => '已完成',
                        2 => '解冻中',
                        3 => '冻结中'
                    ];
                    $row['point_state'] = $pointStateMap[$row['point_state']] ?? '';

                    // 处理积分来源
                    $sourceMap = [
                        1 => '直推',
                        2 => '间推',
                        3 => '购物',
                        9 => '手动'
                    ];
                    $row['source'] = $sourceMap[$row['source']] ?? '';
                }
                return $rows;
            });

            // 隐藏操作按钮
            $grid->disableActions(); // 隐藏所有操作按钮
            $grid->disableCreateButton(); // 隐藏新增按钮
            $grid->disableDeleteButton(); // 隐藏删除按钮

        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new UserPointChangeRec(), function (Show $show) {
            $show->field('id', 'ID');
            $show->field('user_id', '用户ID');
            $show->field('order_id', '关联订单号');
            $show->field('point_amount', '积分数量');
            $show->field('change_type', '增减类别')->using([1 => '增加', 0 => '减少']);
            $show->field('point_state', '积分状态')->using(UserPointChangeRec::pointState);
            $show->field('freeze_end_date', '解冻结束日期');
            $show->field('source', '积分来源')->using(UserPointChangeRec::source);
            $show->field('memo', '备注');
            $show->field('created_at', '创建时间');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new UserPointChangeRec(), function (Form $form) {


            $form->display('id', 'ID');
            $form->text('user_id', '用户ID')->required();
            $form->text('order_id', '关联订单号');
            $form->number('point_amount', '积分数量')->required();
            $form->radio('change_type', '增减类别')
                ->options([1 => '增加', 0 => '减少'])
                ->default(1);
            $form->select('point_state', '积分状态')
                ->options(UserPointChangeRec::pointState)
                ->default(1);
            $form->datetime('freeze_end_date', '解冻结束日期');
            $form->select('source', '积分来源')
                ->options(UserPointChangeRec::source)
                ->default(1);
            $form->text('memo', '备注');
            //$form->display('created_at', '创建时间');
        });
    }


    /**
     * 积分统计方法
     * 
     * @param string $period 统计周期 (day/month/quarter/year)
     * @param string|null $date 指定日期 (格式: Y-m-d)
     * @return array
     */
    public function pointStatistics($period = 'day', $date = null)
    {
        $query = UserPointChangeRec::query();

        switch ($period) {
            case 'day':
                $query->whereDate('created_at', $date ?? date('Y-m-d'));
                break;
            case 'month':
                $query->whereMonth('created_at', $date ?? date('m'));
                break;
            case 'quarter':
                $quarter = $date ? ceil(date('m', strtotime($date)) / 3) : ceil(date('m') / 3);
                $startMonth = (($quarter - 1) * 3) + 1;
                $endMonth = $quarter * 3;
                
                $startDate = date('Y-m-d', strtotime(date('Y') . '-' . $startMonth . '-01'));
                $endDate = date('Y-m-t', strtotime(date('Y') . '-' . $endMonth . '-01'));
                
                $query->whereBetween('created_at', [$startDate, $endDate]);
                break;
            case 'year':
                $query->whereYear('created_at', $date ?? date('Y'));
                break;
        }

        $increase = (clone $query)->where('change_type', 1)->sum('point_amount');
        $decrease = (clone $query)->where('change_type', 0)->sum('point_amount');

        return [
            'period' => $period,
            'date' => $date ?? ($period === 'day' ? date('Y-m-d') : ($period === 'month' ? date('Y-m') : ($period === 'quarter' ? 'Q' . ceil(date('m') / 3) : date('Y')))),
            'increase' => $increase,
            'decrease' => $decrease,
            'net_change' => $increase - $decrease
        ];
    }
}
