<?php

namespace App\Admin\Controllers;

use App\Admin\Renderable\OrderGoodsList;
use App\Models\OrderGoods;
use App\Models\OrderInfo;
use App\Models\City;
use App\Models\Store;
use App\Models\SystemCity;
use App\Models\UserAddress;
use App\Models\Merchant;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Widgets\Card;
use Dcat\Admin\Http\Controllers\AdminController;
use App\Admin\Forms\VerifierCodeForm;
use App\Admin\Forms\ShippingForm;
use Dcat\Admin\Admin;
use App\Http\Controllers\Api\OrderController;
use Illuminate\Support\Facades\DB;


class OrderInfoController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(OrderInfo::with(['user', 'merchant']), function (Grid $grid) {
            $grid->model()->orderBy('id', 'DESC');
            $grid->column('id')->sortable();
            $grid->column('order_sn', '订单号')->width(80);

            // 添加用户昵手机号
            $grid->column('mobile', '手机号')->display(function ($name) {
                return "<a href='/user?id={$this->user_id}' >{$this->mobile}</a>";
            });

            $grid->column('is_div', '是否分账')->display(function ($val) {
                return $val ? '是' : '否';
            });

            $grid->column('goods', '商品信息')->expand(function (Grid\Displayers\Expand $expand) {
                $expand->button('查看');
                return OrderGoodsList::make(['order_id' => $this->id]);
            });

            //1：微信，2：积分 3：混合
            $grid->column('pay_type', '支付类型')->display(function ($val) {
                $types = [
                    1 => '微信',
                    2 => '积分',
                    3 => '混合'
                ];
                return $types[$val] ?? '未知支付方式';
            });
            $grid->column('goods_amount', '订单总金额');
            $grid->column('order_amount', '实际支付');
            $grid->column('point_amount', '积分抵扣');

            $grid->column('merchant.province_id', '商家及收货信息')->display('点击查看')->modal(function ($modal) {
                //设置弹窗标题
                $modal->title('商家及收货信息');
                $company = $this->merchant->name ?? '';
                $content = "<strong>商家信息：</strong>  " . $company . "<br/><br/>";

                // 商家地址
                $province_id = $this->merchant->province_id ?? '';
                if ($province_id) {
                    $proObj = City::where("city_id", $province_id)->first();
                    $cityObj = City::where("city_id", $this->merchant->city_id)->first();
                    $proname = $proObj->name ?? '';
                    $cityname = $cityObj->name ?? '';
                    $content .= "<strong>商家地址：</strong>  " . $proname . " " . $cityname . "<br/><br/>";
                }

                // 收货方式 1:代收点 2：送货上门
                $deliveryTypeText = $this->delivery_type == '1' ? '快递代收点' : '送货上门';
                $content .= "<strong>收货方式：</strong>  " . $deliveryTypeText . "<br/><br/>";
                if ($this->delivery_type == '1') {

                    // 快递代收点信息
                    if ($this->store_id) {
                        $storeObj = Store::where('id', $this->store_id)->first();
                        $content .= "<strong>快递代收点：</strong>  " . ($storeObj->address ?? '') . "<br/><br/>";
                    }
                } else {

                    // 送货上门信息
                    if ($this->area) {
                        $content .= "<strong>送货地址：</strong>  " . $this->area . "(" . $this->address . ")" . "<br/><br/>";
                    }
                }

                // 核销码
                if ($this->verification_code) {
                    $content .= "<strong>核销码：</strong>  " . $this->verification_code . "<br/><br/>";
                }

                // 核销码图片
                if ($this->verification_code_img) {
                    $domain = env('IMAGE_URL'); // 获取配置中的域名
                    $content .= "<strong>核销码:</strong><br/>" . '<img src="' . $domain . $this->verification_code_img . '" style="max-width:100px;max-height:100px;" />' . "<br/><br/>";
                }

                // 核销员信息
                if ($this->verifier) {
                    $content .= "<strong>核销员：</strong>  " . $this->verifier . "<br/><br/>";
                }
                if ($this->verification_at) {
                    $content .= "<strong>核销时间：</strong>  " . $this->verification_at . "<br/><br/>";
                }

                $card = new Card(null, $content);
                return "<div style='padding:10px 10px 0;width:100%;'>$card</div>";
            });
            $grid->column('order_status', '状态')->display(function ($val) {
                $options = OrderInfo::STATUS_OPTIONS;
                return $options[$val];
            })->label([
                0 => 'primary',   // 待付款 - 蓝色
                1 => 'warning',   // 待发货 - 橙色
                2 => 'info',      // 待到货 - 青色
                3 => 'pink',      // 待评价 - 粉色
                4 => 'success',   // 已完成 - 绿色
                7 => 'danger',    // 已取消 - 红色
                8 => 'dark'       // 已退款 - 黑色
                // - purple (紫色)
                // - teal (蓝绿色)
                // - indigo (靛蓝色)
                // - yellow (黄色)
                // - gray (灰色)
            ])->if(function ($column) {
                return $column->getValue() == 1;
            })->display('点击生成核销码')->modal(function (Grid\Displayers\Modal $modal) {
                // 标题
                $modal->title('生成核销码');
                // 自定义图标
                $modal->icon('feather icon-edit');
                // 传递当前行字段值
                return VerifierCodeForm::make()->payload(['id' => $this->id]);
            });
            $grid->column('created_at', '下单时间');

            //$grid->column('updated_at')->sortable();

            //$grid->disableActions();
            $grid->disableDeleteButton();
            $grid->disableEditButton();
            $grid->disableCreateButton();
            $grid->disableViewButton();

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->equal('order_sn', '订单号')->width(3);
                $filter->like('mobile', '手机号')->width(3);
                $filter->equal('merchant.id', '所属商家')->select(function () {
                    return Merchant::all()->pluck('name', 'id');
                })->width(3);
                //订单状态 - 修改为多选
                $filter->in('order_status', '订单状态')
                    ->multipleSelect(OrderInfo::STATUS_OPTIONS)
                    ->width(3);

                //是否分账
                $filter->equal('is_div', '是否分账')->select(['1' => '是', '0' => '否'])->width(3);

                //支付类型
                $filter->in('pay_type', '支付类型')
                    ->multipleSelect([
                        1 => '微信',
                        2 => '积分',
                        3 => '混合'
                    ])
                    ->width(3);

                // 添加时间范围快捷筛选
                $filter->scope('today', '今日')->whereDate('created_at', date('Y-m-d'));
                $filter->scope('month', '本月')->whereMonth('created_at', date('m'));
                $filter->scope('quarter', '本季度')->where(function ($query) {
                    $month = date('m');
                    $year = date('Y');
                    
                    if ($month >= 1 && $month <= 3) {
                        // 第一季度 (1-3月)
                        $start = date('Y-m-d', strtotime('first day of January'));
                        $end = date('Y-m-d', strtotime('last day of March'));
                    } elseif ($month >= 4 && $month <= 6) {
                        // 第二季度 (4-6月)
                        $start = date('Y-m-d', strtotime('first day of April'));
                        $end = date('Y-m-d', strtotime('last day of June'));
                    } elseif ($month >= 7 && $month <= 9) {
                        // 第三季度 (7-9月)
                        $start = date('Y-m-d', strtotime('first day of July'));
                        $end = date('Y-m-d', strtotime('last day of September'));
                    } else {
                        // 第四季度 (10-12月)
                        $start = date('Y-m-d', strtotime('first day of October'));
                        $end = date('Y-m-d', strtotime('last day of December'));
                    }
                    
                    $query->whereBetween('created_at', [$start, $end]);
                });
                $filter->scope('year', '本年')->whereYear('created_at', date('Y'));

                $filter->between('created_at', '自定义时间范围')->datetime()->width(4);
            });


            $titles = [
                'id' => 'ID',
                'order_sn' => '订单号',
                'mobile' => '手机号',
                'goods_name' => '商品名称',
                'goods_attr' => '商品规格',
                'goods_number' => '商品数量',
                'goods_price' => '商品单价',
                'goods_amount' => '订单金额',
                'order_amount' => '实际支付',
                'point_amount' => '积分抵扣',
                'order_status' => '订单状态',
                'created_at' => '下单时间'
            ];

            // 添加导出按钮
            $grid->export($titles)->rows(function ($rows) {
                foreach ($rows as $index => &$row) {
                    // 获取商品信息
                    $goods = OrderGoods::where('order_id', $row['id'])->get();
                    $goodsInfo = [];
                    foreach ($goods as $good) {
                        $goodsInfo[] = [
                            'goods_name' => $good->goods_name,
                            'goods_attr' => $good->goods_attr,
                            'goods_number' => $good->goods_number,
                            'goods_price' => $good->goods_price
                        ];
                    }
                    $row['goods_name'] = implode("\n", array_column($goodsInfo, 'goods_name'));
                    $row['goods_attr'] = implode("\n", array_column($goodsInfo, 'goods_attr'));
                    $row['goods_number'] = implode("\n", array_column($goodsInfo, 'goods_number'));
                    $row['goods_price'] = implode("\n", array_column($goodsInfo, 'goods_price'));

                    // 其他字段处理
                    $row['address_id'] = ($row['address_id'] == '') ? '是' : '否';
                    $row['order_status'] = OrderInfo::STATUS_OPTIONS[$row['order_status']];
                }
                return $rows;
            });

            // 添加退款操作按钮
            $grid->actions(function (Grid\Displayers\Actions $actions) {
                //只对特定状态的订单显示退款按钮
                if ($this->is_div == 0) {
                    $actions->append('<a href="javascript:void(0)" class="refund-order" data-id="' . $this->id . '"><i class="feather icon-refresh-ccw"></i>  退款</a>');
                }
            });

            // 添加退款操作的JS代码
            Admin::script(<<<JS
            $('.refund-order').click(function() {
                var id = $(this).data('id');
                Dcat.confirm('确定要给该笔订单退款吗？支付金额会原路返回。', null, function() {
                    Dcat.loading();
                    $.ajax({
                        url: '/order-refund',
                        type: 'POST',
                        data: {
                            id: id,
                            _token: Dcat.token
                        },
                        success: function(data) {
                            Dcat.loading(false);
                            if (data.status) {
                                Dcat.success(data.message);
                                Dcat.reload();
                            } else {
                                Dcat.error(data.message);
                            }
                        },
                        error: function() {
                            Dcat.loading(false);
                            Dcat.error('请求失败，请重试');
                        }
                    });
                });
            });
            JS);
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new OrderInfo(), function (Show $show) {
            $show->field('id');
            $show->field('order_sn', '订单号');
            $show->field('verification_code', '核销码');
            // $show->field('order_status')->display(function ($val) {
            //     $options = OrderInfo::STATUS_OPTIONS;
            //     return $options[$val];
            // });
            $show->field('created_at', '下单时间');
            //$show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new OrderInfo(), function (Form $form) {
            //$form->display('id');
            $form->display('order_sn', '订单号');
            $infoObj = $form->model(); // 获取当前数据
            //$form->text('remark', '备注(后台用)');
            //订单状态 -1:全部 0:待付款 1:待到货 2:待领取 3: 待评价 4:已完成  7:已取消 8:已退款
            if ($form->model()->order_status == 2) {
                $options = [
                    3 => '待评价'
                ];

                $form->select('order_status', '订单状态')->options($options);
            }
            // 禁用选项2
            //$options[2] = ['text' => '选项2', 'disable' => true];


            // $form->switch('pay_status')->default(0);
            // $form->switch('shipping_status', '发货状态')->default(0);
            // $form->text('order_amount', '订单应付款');
            // $form->display('created_at');
            // $form->display('updated_at');
            $form->disableEditingCheck();
            $form->disableViewCheck();
            $form->disableDeleteButton();
            $form->disableViewButton();
        });
    }



    //管理员退款
    /**
     * 管理员退款功能
     * 功能：处理订单退款操作，包括退款金额原路返回和库存恢复
     * 参数：无显式参数，通过request获取订单ID
     * 返回值：json格式，包含操作状态和消息
     * 异常：捕获数据库操作异常并回滚
     * 使用示例：前端通过AJAX调用/order-refund接口
     * 注意事项：1.仅对特定状态订单可退款 2.使用事务保证数据一致性 3.记录操作日志
     */
    public function OrderAdminRefund()
    {
        $id = request('id');
        $orderObj = OrderInfo::find($id);
        if (!$orderObj) {
            return response()->json(['status' => false, 'message' => '订单不存在']);
        }

        if ($orderObj->is_div == 0) {
            return OrderController::canceOrderFunc($orderObj);
        } else {
            return response()->json(['status' => false, 'message' => '已分账，不允许退款！']);
        }

        //return response()->json(['status' => true, 'message' => '退款成功']);
    }
}
