<?php

namespace App\Admin\Controllers;

use App\Admin\Renderable\OrderGoodsList;
use App\Models\OrderGoods;
use App\Models\OrderInfo;
use App\Models\City;
use App\Models\Store;
use App\Models\SystemCity;
use App\Models\UserAddress;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use App\Admin\Forms\VerifierCodeForm;


class OrderInfoController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(OrderInfo::with(['user', 'merchant']), function (Grid $grid) {
            $grid->model()->orderBy('created_at', 'DESC');
            $grid->column('id')->sortable();
            $grid->column('order_sn', '订单号')->width(80);
            $grid->column('mobile', '手机号');

            $grid->column('goods', '商品信息')->expand(function (Grid\Displayers\Expand $expand) {
                $expand->button('查看');
                return OrderGoodsList::make(['order_id' => $this->id]);
            });
            $grid->column('goods_amount', '订单总金额');
            $grid->column('merchant.name', '所属商家');
            $grid->column('merchant.province_id', '所在省市')->display(function ($val) {
                if ($val) {
                    $proObj = City::where("city_id", $val)->first();
                    $cityObj = City::where("city_id", $this->merchant->city_id)->first();
                    $proname = $proObj->name ?? '';
                    $cityname = $cityObj->name ?? '';
                    return $proname . " " . $cityname;
                } else {
                    return '';
                }
            });

            $grid->column('store_id', '快递代收点')->display(function ($val) {
                $res = null;
                if ($val) {
                    $obj = Store::where('id', $val)->first();
                    $res = $obj->address;
                }
                return $res;
            });
            $grid->column('address_id', '送货上门')->display(function ($val) {
                $res = '';
                if ($this->area) {
                    $res = $this->area . "(" . $this->address . ")";
                }
                return $res;
            });
            $grid->column('created_at', '下单时间');
            $grid->column('order_status', '状态')->display(function ($val) {
                $options = OrderInfo::STATUS_OPTIONS;
                return $options[$val];
            })->if(function ($column) {
                return $column->getValue() == 1;
            })->display('点击生成核销码')->modal(function (Grid\Displayers\Modal $modal) {
                // 标题
                $modal->title('生成核销码');
                // 自定义图标
                $modal->icon('feather icon-edit');
                // 传递当前行字段值
                return VerifierCodeForm::make()->payload(['id' => $this->id]);
            });
            $grid->column('verification_code', '核销码')->limit(10);
            $grid->column('verifier', '核销员');
            $grid->column('verification_at', '核销时间');
            //$grid->column('updated_at')->sortable();

            //$grid->disableActions();
            $grid->disableCreateButton();
            $grid->disableViewButton();
            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->equal('order_sn', '订单号')->width(3);
                $filter->like('mobile', '手机号')->width(3);
                $filter->like('merchant.name', '所属商家')->width(3);
                //订单状态
                $filter->equal('order_status', '订单状态')->select(OrderInfo::STATUS_OPTIONS)->width(3);
                $filter->equal('merchant.province_id', '省份')->select(City::where('parent_id', 0)->get()->pluck('name', 'city_id'))->load('merchant.city_id', '/city')->width(3);
                // 城市
                $filter->equal('merchant.city_id', '城市')->select()->width(3);
                $filter->between('created_at', '创建时间')->datetime()->width(4);
            });
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new OrderInfo(), function (Show $show) {
            $show->field('id');
            $show->field('order_sn', '订单号');
            $show->field('verification_code', '核销码');
            // $show->field('order_status')->display(function ($val) {
            //     $options = OrderInfo::STATUS_OPTIONS;
            //     return $options[$val];
            // });
            $show->field('created_at', '下单时间');
            //$show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new OrderInfo(), function (Form $form) {
            //$form->display('id');
            $form->display('order_sn', '订单号');
            $infoObj = $form->model(); // 获取当前数据
            //$form->text('remark', '备注(后台用)');
            //订单状态 -1:全部 0:待付款 1:待到货 2:待领取 3: 待评价 4:已完成  7:已取消 8:已退款
            if ($form->model()->order_status == 2) {
                $options = [
                    3 => '待评价'
                ];

                $form->select('order_status', '订单状态')->options($options);
            }
            // $options = [
            //     1 => '选项1',
            //     2 => '选项2',
            //     'value2' => ['text' => '选项二', 'disabled' => true], // 这个选项是只读的
            // ];

            // 禁用选项2
            //$options[2] = ['text' => '选项2', 'disable' => true];


            // $form->switch('pay_status')->default(0);
            // $form->switch('shipping_status', '发货状态')->default(0);
            // $form->text('order_amount', '订单应付款');
            // $form->display('created_at');
            // $form->display('updated_at');
            $form->disableEditingCheck();
            $form->disableViewCheck();
            $form->disableDeleteButton();
            $form->disableViewButton();
        });
    }
}
