<?php

namespace App\Admin\Controllers;

use App\Models\MerchantStore;
use App\Models\Merchant;
use App\Models\Good;
use App\Models\MerchantGoodSku;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use App\Models\City;
use App\Models\Store;
use Illuminate\Support\Facades\DB;
use Dcat\Admin\Http\Controllers\AdminController;
use App\Admin\Actions\MerchantStore as MerchantAddStore;
use App\Admin\Actions\MerchantSubStore;
use Dcat\Admin\Admin;
use Dcat\Admin\Layout\Content;
use Illuminate\Http\Request;
use App\Admin\Renderable\MerchantStoreSkuList;

class MerchantGoodsStoreController extends AdminController
{

    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(MerchantStore::with(['goods', 'merchant']), function (Grid $grid) {
            $grid->model()->orderBy('created_at', 'DESC');
            $grid->column('id')->sortable();
            $grid->column('goods_sn', '商品编号');
            $grid->column('goods.goods_name', '商品名称');
            $grid->column('goods.cover_img', '商品图片')->image('', 100, 100);
            $grid->column('attr', '商品规格')->expand(function (Grid\Displayers\Expand $expand) {
                $expand->button('查看');
                return MerchantStoreSkuList::make(['mgs_id' => $this->id]);
            })->width(100);
            $grid->column('store', '库存')->display(function ($val) {
                return MerchantGoodSku::where(['mgs_id' => $this->id])->sum('stock');
            });
            $grid->column('merchant.name', '商家名称');
            $grid->column('merchant.contacts', '联系人');
            //$grid->column('created_at');
            //$grid->column('updated_at')->sortable();

            $grid->disableViewButton();
            //$grid->disableCreateButton();

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->like('goods.goods_name', '商品名称')->width(3);
                $filter->like('merchant.name', '商家名称')->width(3);
                $filter->like('merchant.contacts', '联系人')->width(3);
            });

            $grid->actions(function (Grid\Displayers\Actions $actions) {
                $actions->append(new MerchantAddStore('增加库存', $actions->row->goods_id));
                $actions->append(new MerchantSubStore('减少库存', $actions->row->goods_id));
            });
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new MerchantStore(), function (Show $show) {
            $show->field('id');
            $show->field('name');
            $show->field('contacter');
            $show->field('phone');
            $show->field('created_at');
            $show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        $form = Form::make(new MerchantStore(), function (Form $form) {
            $form->display('id');
            $form->select('merchant_id', '商家名称')->options(Merchant::whereNull('deleted_at')->get()->pluck('name', 'id'))->rules('required'); //->load('merchat_id', '/get-merchat-list');
            $form->select('goods_id', '添加商品')->options((new Good())->limit(100)->get()->pluck('goods_name', 'id'))->rules('required'); //->load('merchat_id', '/get-merchat-list');


            $form->disableCreatingCheck();
            $form->disableEditingCheck();
            $form->disableViewCheck();
            $form->disableDeleteButton();
            $form->disableViewButton();
        });

        $form->submitted(function (Form $form) {
            $merchant_id = isset($_POST['merchant_id']) ? $_POST['merchant_id'] : 0;
            $goods_id = isset($_POST['goods_id']) ? $_POST['goods_id'] : 0;

            $count = MerchantStore::where(['goods_id' => $goods_id, 'merchant_id' => $merchant_id])->count();
            if ($count) {
                $form->responseValidationMessages('goods_id', "该商品已添加！");
            }
        });

        $form->saved(
            function (Form $form, $result) {
                $merchant_gid = $form->getKey();
                $msObj = MerchantStore::find($merchant_gid);
                if ($msObj) {
                    $goods_id = $msObj->goods_id;
                    $merchant_id = $msObj->merchant_id;
                    $goodsObj = Good::where("id", $goods_id)->first();
                    $merchatObj = Merchant::where("id", $merchant_id)->first();
                    $params = [
                        'goods_sn' => $goodsObj->goods_sn,
                        'goods_name' => $goodsObj->goods_name,
                        'merchant_name' => $merchatObj->name,
                        'contacts' => $merchatObj->contacts,
                    ];
                    DB::table('merchat_goods_store')->where("id", $merchant_gid)->update($params);
                }
            }
        );

        return $form;
    }

    public function getList(Request $request)
    {
        $mId = $request->get('q');

        return Store::where('merchant_id', $mId)->get(['id', 'title as text']);
    }
}
