<?php

namespace App\Admin\Controllers;

use App\Command\Log;
use App\Models\Good;
use App\Models\GoodSku;
use App\Models\Category;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Widgets\Card;
use Dcat\Admin\Widgets\Modal;
use App\Admin\Forms\ImportExcel;
use App\Admin\Actions\DownloadTemplateExcel;
use Illuminate\Support\Facades\DB;
use App\Admin\Renderable\RenderGoodSku;
use App\Admin\Repositories\GoodSku as SkuRepos;
use App\Admin\Actions\GoodsExcel;
use Illuminate\Http\Request;
use App\Admin\Extensions\GoodsExport;
use Maatwebsite\Excel\Facades\Excel;

class GoodController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new Good(), function (Grid $grid) {
            $grid->model()->orderBy('created_at', 'DESC');
            $grid->column('id')->sortable();
            $grid->column('goods_sn', '商品编号')->width(80);
            $grid->column('goods_name')->limit(15)->width(250);
            $grid->column('cover_img', '商品图片')->image('', 100, 100);
            $grid->column('sku', '规格信息')->expand(function (Grid\Displayers\Expand $expand) {
                $expand->button('查看');
                return RenderGoodSku::make(['goods_id' => $this->id]);
            });
            $grid->column('goods_desc', '内容详情')->if(function ($column) {
                return $column->getValue();
            })->display('点击查看')->modal(function ($modal) {
                //设置弹窗标题
                $modal->title('内容详情');

                $card = new Card(null, $this->goods_desc);

                return "<div style='padding:10px 10px 0;width:100%;'>$card</div>";
            })->else(function ($column) {
                return '';
            });
            $grid->column('intro_desc', '购买须知')->if(function ($column) {
                return $column->getValue();
            })->display('点击查看')->modal(function ($modal) {
                //设置弹窗标题
                $modal->title('内容详情');

                $card = new Card(null, $this->intro_desc);

                return "<div style='padding:10px 10px 0;width:100%;'>$card</div>";
            })->else(function ($column) {
                return '';
            });
            $grid->column('sort', '排序');
            $grid->column('is_hot', '是否推荐')->display(function ($val) {
                return ($val == 1) ? '是' : '否';
            });
            $grid->column('is_show', '状态')->switch('', true);
            //$grid->column('updated_at')->sortable();

            $grid->disableViewButton();
            $grid->actions(function (Grid\Displayers\Actions $actions) {
                // 假设我们根据模型的 status 字段来控制编辑按钮的显示
                if ($actions->row->is_show == 1) {
                    //则不显示编辑按钮
                    $actions->disableEdit();
                }
            });

            $grid->filter(function (Grid\Filter $filter) {
                // 更改为 panel 布局
                $filter->panel();
                $filter->like('goods_name')->width(3);
                $filter->equal('is_hot', '是否推荐')->select(['1' => '是', '0' => '否'])->width(3);
                $filter->equal('is_show', '上架状态')->select(['1' => '是', '0' => '否'])->width(3);
            });
        });
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     *
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, Good::with(['category']), function (Show $show) {
            $show->field('id');
            $show->field('goods_name');
            $show->field('category.title', '分类名称');
            $show->field('product_no');
            $show->field('material');
            $show->field('product_no');
            $show->field('box_size');
            $show->relation('gattr', function ($model) {
                $grid = new Grid(new SkuRepos());

                $grid->model()->where('goods_id', $model->id);
                // 设置路由
                $grid->setResource('gattr');

                $grid->id();
                $grid->content()->limit(10);
                $grid->attr_val();
                $grid->attr_sn();
                $grid->created_at();
                $grid->updated_at();

                // $grid->filter(function ($filter) {
                //     $filter->like('content')->width('300px');
                // });

                return $grid;
            });

            $show->field('created_at');
            $show->field('updated_at');
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        $form = Form::make(new Good(), function (Form $form) {
            $form->display('id');
            $form->tab('基本信息', function (Form $form) {
                $form->text('goods_name', '商品名称')->required();
                $form->text('goods_sn', '商品编号');
                $form->text('first_commission', '直推佣金比例')->help("比例如：30");
                $form->text('second_commission', '间推佣金比例')->help("比例如：10");
                $form->text('merchant_commission', '商家佣金比例')->help("比例如：10");
                $form->text('sale', '销量');
                $form->text('high_opinion', '好评率');
                //$form->text('box_size');
                $form->image('cover_img')
                    ->accept('jpg,jpeg,png')
                    ->maxSize(4096)
                    ->url('upload/goods')
                    ->help('仅支持jpg、jpeg、png格式图片上传（320px * 320px）')
                    ->autoUpload();
                $form->multipleImage('carousel', '产品图')
                    ->accept('jpg,jpeg,png')
                    ->maxSize(51200)
                    ->url('upload/goods')
                    ->help('仅支持jpg、jpeg、png格式图片上传（尺寸 750px*750px）')
                    ->limit(5)
                    ->autoUpload()->saveAsJson();
                $form->switch('is_show', '上架状态')->default(1);
                $form->switch('is_hot', '是否推荐')->default(0);
                $form->text('sort', '排序')->default(0)->help('越大越靠前');
                $form->disableCreatingCheck();
                $form->disableEditingCheck();
                $form->disableViewCheck();
                $form->disableDeleteButton();
                $form->disableViewButton();
            })->tab('规格', function (Form $form) {
                $sku = $form->model()->sku;
                //$str = '{"attrs":{"颜色":["红色","蓝色"],"大小":["20"]},"sku":[{"颜色":"红色","大小":"20","pic":"图片","stock":"106","price":"100","column1":"5","column2":""},{"颜色":"蓝色","大小":"20","pic":"","stock":"89","price":"90","column1":"51","column2":""}]}';
                $sku_params = [
                    [
                        'name'    => '库存', // table 第一行 title 
                        'field'   => 'stock', // input 的 field_name 名称
                        'default' => '', // 默认值
                    ],
                ];

                $form->sku('sku', json_encode($sku_params))->display(true)->customFormat(function ($value) use ($form, $sku) {

                    return $sku;
                })->required();
            })->tab('文字标签', function (Form $form) {
                $fieldArr = [
                    'field1' => '',
                    'field2' => '',
                    'field3' => '',
                    'field4' => '',
                    'field5' => '',
                ];
                $form->keyValue('tags', '标签')
                    ->setKeyLabel('键 (field开头拼上数字,商品列表页filed1-10,商品详情页field11-30)')
                    ->setValueLabel('值')->saving(function ($v) {
                        // 转化为json格式存储
                        return json_encode($v);
                    });
                //->disableCreateButton();
                //->disableDelete();
            })->tab('详情', function (Form $form) {

                $form->editor('goods_desc');
            })->tab('购买须知', function (Form $form) {

                $form->editor('intro_desc');
            });
        });

        //副表保存规格
        $form->saved(function (Form $form, $result) {
            $goods_id = $form->getKey();
            //是否操作规格数据(编辑时)
            $skuexist = GoodSku::where("goods_id", $goods_id)->count();
            //Log::add('规格数据', $skuArr);
            if (isset($_POST['sku'])) {
                $skuStr = $_POST['sku'];
                $skuArr = json_decode($skuStr, true);
                $attrs = isset($skuArr['attrs']) ? $skuArr['attrs'] : [];
                $attrsKey = $attrs ? array_keys($attrs) : [];
                if ($skuexist == 0) { //新增
                    $skuArr['sku'] = self::addGoodSkuData($attrsKey, $skuArr['sku'], $goods_id);
                } else {
                    //编辑
                    $skuArr['sku'] = self::editGoodSkuData($attrsKey, $skuArr['sku'], $goods_id);
                }
                $skuCon = json_encode($skuArr, JSON_UNESCAPED_UNICODE);
                DB::table('li_goods')->where("id", $goods_id)->update(['sku' => $skuCon]);
            }
        });
        $form->submitted(function (Form $form) {
            $goods_id = $form->getKey();
            $is_show = isset($_POST['is_show']) ? $_POST['is_show'] : 0;
            $skuStr = isset($_POST['sku']) ? $_POST['sku'] : '';
            $skuArr = json_decode($skuStr, true);
            $skuVal = isset($skuArr['sku']) ? $skuArr['sku'] : [];
            if (isset($_POST['sku']) && !$skuVal && $is_show) {
                $form->responseValidationMessages('sku', "商品规格不为空");
            }
        });
        return $form;
    }
    public static function addGoodSkuData($attrsKey, $sku, $gid)
    {
        foreach ($sku as $kk => $vv) {
            $attr_name = [];
            foreach ($attrsKey as $val) {
                array_push($attr_name, $vv[$val]);
            }
            utf8_array_asort($attr_name);
            $attr_val = join("、", $attr_name);
            $attr_sn =  md5($attr_val);
            $goodSkuObj = new GoodSku();
            $goodSkuObj->goods_id = $gid;
            $goodSkuObj->attr_val = $attr_val;
            $goodSkuObj->attr_sn = $attr_sn;
            $goodSkuObj->market_price = $vv['market_price'];
            $goodSkuObj->cg_price = $vv['cg_price'];
            $goodSkuObj->stock = $vv['stock'];
            $goodSkuObj->pic = $vv['pic'];
            $goodSkuObj->save();
            //规格标识
            $sku[$kk]['attr_sn'] = $attr_sn;
        }
        return $sku;
    }
    public static function editGoodSkuData($attrsKey, $sku, $gid)
    {
        $existAttrSn = []; //规格数组
        $editAttrSn = []; //已编辑存在的规格
        $glist = GoodSku::where("goods_id", $gid)->get()->toArray();
        foreach ($glist as $key => $val) {
            array_push($existAttrSn, $val['attr_sn']);
        }

        foreach ($sku as $kk => $vv) {
            $attr_name = [];
            foreach ($attrsKey as $val) {
                array_push($attr_name, $vv[$val]);
            }
            utf8_array_asort($attr_name);
            $attr_val = join("、", $attr_name);
            $attr_sn = md5($attr_val);
            if (in_array($attr_sn, $existAttrSn)) {
                $goodSkuObj = GoodSku::where(['attr_sn' => $attr_sn, 'goods_id' => $gid])->first();
                $editAttrSn[] = $attr_sn;
            } else {
                $goodSkuObj = new GoodSku();
                $goodSkuObj->goods_id = $gid;
            }
            $goodSkuObj->attr_val = $attr_val;
            $goodSkuObj->pic = str_replace(env('IMAGE_URL'), '', $vv['pic']);
            $goodSkuObj->attr_sn = $attr_sn;
            $goodSkuObj->market_price = $vv['market_price'];
            $goodSkuObj->cg_price = $vv['cg_price'];
            $goodSkuObj->stock = $vv['stock'];
            $goodSkuObj->save();

            //规格标识
            $sku[$kk]['attr_sn'] = $attr_sn;
        }
        //取数组差集
        $noAttrSn = array_diff($existAttrSn, $editAttrSn);
        if ($noAttrSn) {
            foreach ($noAttrSn as $k => $v) {
                DB::table('li_goods_sku')->where(['goods_id' => $gid, 'attr_sn' => $v])->delete();
            }
            //清除商家商品记录
            DB::table('merchant_store_sku')->where(['goods_id' => $gid])->delete();
        }

        return $sku;
    }

    /**
     * 导出
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        $filename = $request->get('filename');

        $param = json_decode($request->get('param'));

        ob_end_clean();
        return Excel::download(new GoodsExport($param), $filename . '.xlsx');
    }
}
