<?php

namespace App\Store\Controllers;

use App\Command\Log;
use App\Command\Tools; // 引入 Tools 类
use App\Models\Merchant;
use App\Models\OrderDivideRecord;
use Dcat\Admin\Admin;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Show;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Widgets\Card;
use Dcat\Admin\Layout\Content;
use Dcat\Admin\Widgets\Box;
use App\Http\Controllers\Api\StoreAdminUsersController;

class OrderDivideRecordController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(OrderDivideRecord::with(['users']), function (Grid $grid) {
            $merchant_id = Admin::user()->merchant_id;
            $total_revenue = $balance = $cashout = 0;
            if ($merchant_id) {
                $merObj = Merchant::where('id', $merchant_id)->first();
                
                // 获取分佣记录
                $orderDivideRecord = OrderDivideRecord::where(['um_id' => $merchant_id])
                    ->select('divide_price', 'is_div', 'created_at')
                    ->orderBy('created_at', 'desc')
                    ->get();
            
                // 使用getCommissionAmount计算未分账金额
                $balance01 = StoreAdminUsersController::getCommissionAmount(1, $merchant_id);
                
                // 计算冻结中金额(状态为0的未绑卡金额)
                $balance02 = $orderDivideRecord->where('is_div', 0)->sum('divide_price');
                
                // 计算已提现金额(状态为1的已分账金额)
                $cashout = number_format($orderDivideRecord->where('is_div', 1)->sum('divide_price'), 2);
                
                // 计算冻结中金额 = 未分账金额 + 已冻结金额
                $balance = number_format($balance01 + $balance02, 2);
                
                // 总收益 = 冻结中 + 已提现
                $total_revenue = number_format($balance + $cashout, 2);
            
                Log::add('balance金额', $balance);
                Log::add('cashout金额', $cashout);
            }
            $grid->header(function () use ($total_revenue, $balance, $cashout) {
                $box = new Box('统计信息', '商户收益统计');

                // 自定义统计框样式
                $box->style('info');
                $content = '<div style="height: 135px;display: flex;align-items: center;text-align: center;">
                    <div style="flex: 1;display: flex;flex-direction: column;border-right: 1px solid #b9c3cd;">
                    <h2 style="font-size: 25px;margin-top: 25px;">总收益(元)</h2>
                    <p style="color: orange;font-size: 45px;font-weight: bold;">' . number_format($total_revenue, 2) . '</p>
                    </div>
                    <div style="flex: 1;display: flex;flex-direction: column;border-right: 1px solid #b9c3cd;">
                    <h2 style="font-size: 25px;margin-top: 25px;">冻结中(元)</h2>
                    <p style="color: #3399ff;font-size: 45px;font-weight: bold;">' . $balance . '</p>
                    </div>
                    <div style="flex: 1;display: flex;flex-direction: column;">
                    <h2 style="font-size: 25px;margin-top: 25px;">已提现(元)</h2>
                    <p style="color: #21b978;font-size: 45px;font-weight: bold;">' . $cashout . '</p>
                    </div></div>';
                
                $box->content($content);
                return $box->render();
            });


            $grid->addTableClass(['table-text-center']);

            $grid->model()->where(['sh_type' => 3, 'um_id' => $merchant_id])->orderBy('created_at', 'DESC');
            $grid->column('id')->sortable();
            $grid->column('title', '来源名称')->display(function ($val) {
                return $val ?? '用户取货佣金';
            });
            $grid->column('order_id', '关联订单ID');
            $grid->column('users.phone', '用户手机号');
            $grid->column('divide_price', '金额');
            $grid->column('created_at', '时间');

            // 添加筛选功能
            $grid->filter(function (Grid\Filter $filter) {
                $filter->panel();
                $filter->equal('order_id', '订单ID')->width(3);
                $filter->like('users.phone', '用户手机号')->width(3);
                $filter->between('created_at', '时间范围')->datetime()->width(6);
            });

            // 启用导出功能
            $grid->export()->filename('分佣记录-'.date('Y-m-d'));

            $grid->disableDeleteButton();
            $grid->disableViewButton();
            $grid->disableCreateButton();
            $grid->disableBatchActions();
            $grid->disableRefreshButton();
            $grid->disableActions();
            $grid->disableRowSelector();
            // $grid->filter(function (Grid\Filter $filter) {
            //     $filter->equal('id');
            // });
        });
    }
}
