<?php

namespace App\Models;

use Dcat\Admin\Traits\HasDateTimeFormatter;
use Illuminate\Database\Eloquent\Model;
use App\Command\Log;
use Illuminate\Support\Facades\DB;
use Exception;
use App\Models\User as UserModel;
use App\Models\OrderInfo;

class UserPointChangeRec extends Model
{
    use HasDateTimeFormatter;
    protected $table = 'user_point_change_rec';
    public $timestamps = false;

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    
    public function order()
    {
        return $this->belongsTo(OrderInfo::class, 'order_id');
    }


    //积分来源定义【source】
    public const source = [
        1 => '直推',
        2 => '间推',
        3 => '购物',
        4 => '退款',
        9 => '手动补偿',
    ];

    
    //积分状态定义【point_state】
    public const pointState = [
        1 => '已完成',
        2 => '解冻中',
        3 => '冻结中'
    ];


    /**
     * 用户积分变更记录
     * @param int $user_id 用户ID
     * @param int $point_amount 积分数量
     * @param int $change_type 变更类型：1-增加，0-减少 (默认1)
     * @param int $source 积分来源：1-直推，2-间推，3-购物，4-退款，9-手动补偿 (默认1)
     * @param int $point_state 积分状态：1-已完成，2-解冻中，3-冻结中 (默认1)
     * @param int $order_id 关联订单ID
     * @param string $remark 备注信息
     * @return bool 返回操作结果
     * 
     * 功能说明：
     * 1. 记录用户积分变更情况
     * 2. 仅当point_state=1(已完成)时更新用户余额
     * 3. 记录变更备注信息
     *
     * 使用示例：
     * UserPointChangeRec::pointChangeRecord(1, 100, 1, 1, 1, '订单号：123456');
     * 注意事项：
     * 1. 积分数量必须为正整数
     * 2. 变更类型必须为0或1
     */
    public static function pointChangeRecord($user_id, $point_amount,  $change_type = 1, $source = 1,$point_state = 1,$order_id=0, $remark = '' )
    {
        DB::beginTransaction();
        try {
            // 参数验证
            if ($point_amount <= 0) {
                throw new Exception('积分数量必须大于0');
            }
            if (!in_array($change_type, [0, 1])) {
                throw new Exception('变更类型参数错误');
            }

            // 创建积分变更记录
            DB::table('user_point_change_rec')->insert([
                'user_id' => $user_id,
                'order_id' => $order_id,
                'point_amount' => $point_amount,
                'change_type' => $change_type,
                'point_state' => $point_state,
                'source' => $source,
                'memo' => $remark,
                'created_at' => date('Y-m-d H:i:s')
            ]);

            // 仅当状态为已完成时更新用户余额
            if ($point_state == 1) {
                $user = UserModel::find($user_id);
                if ($change_type == 1) {
                    $user->balance += $point_amount; // 增加积分
                } else {
                    if ($user->balance < $point_amount) {
                        throw new Exception('积分不足');
                    }
                    $user->balance -= $point_amount; // 减少积分
                }
                $user->save();
            }

            DB::commit();
            Log::add('pay', [
                'user_id' => $user_id,
                'point_amount' => $point_amount,
                'change_type' => $change_type,
                'point_state' => $point_state
            ]);
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::add('pay', [
                'user_id' => $user_id,
                'point_amount' => $point_amount,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }


    /**
     * 用户积分解冻
     * @param object $orderObj 订单对象
     * @return bool 返回操作结果
     * 
     * 功能说明：
     * 1. 根据订单ID查找对应的积分变更记录
     * 2. 将冻结中的积分状态更新为解冻中
     * 3. 设置7天后的解冻结束日期
     * 
     * 使用示例：
     * $order = OrderInfo::find(123);
     * UserPointChangeRec::point解冻($order);
     * 
     * 注意事项：
     * 1. 仅处理状态为冻结中(3)的积分记录
     * 2. 解冻后状态变为解冻中(2)
     * 3. 解冻结束日期为当前时间+7天
     */
    public static function pointUnfreezeimg($orderObj)
    {
        DB::beginTransaction();
        try {
            // 查找该订单对应的冻结中积分记录
            $records = DB::table('user_point_change_rec')
                ->where([
                    'order_id' => $orderObj->id,
                    'point_state' => 3, // 冻结中
                    'change_type' => 1   // 增加积分
                ])
                ->get();

            if ($records->isEmpty()) {
                Log::add('point_unfreeze', [
                    'order_id' => $orderObj->id,
                    'msg' => '没有找到可解冻的积分记录'
                ]);
                DB::commit();
                return true;
            }

            // 设置7天后的解冻结束日期
            $freezeEndDate = date('Y-m-d H:i:s', strtotime('+7 days'));

            // 遍历所有记录，更新每条积分记录状态为解冻中
            $totalPointAmount = 0;
            foreach ($records as $record) {
                DB::table('user_point_change_rec')
                    ->where('id', $record->id)
                    ->update([
                        'point_state' => 2, // 解冻中
                        'freeze_end_date' => $freezeEndDate
                    ]);
                $totalPointAmount += $record->point_amount;
            }

            DB::commit();
            Log::add('point_unfreeze', [
                'order_id' => $orderObj->id,
                'total_point_amount' => $totalPointAmount,
                'record_count' => count($records),
                'freeze_end_date' => $freezeEndDate
            ]);
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::add('point_unfreeze_error', [
                'order_id' => $orderObj->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }


    
}
