<?php
namespace App\Command;
use Illuminate\Http\Request;

class Log {
    /**
     * 获取当前请求对象
     * @return Request|null 返回当前请求对象或null
     * 
     * 功能说明：
     * 1. 尝试通过Laravel的request()辅助函数获取当前请求
     * 2. 如果辅助函数不可用，则返回null
     * 
     * 使用示例：
     * $request = Log::getCurrentRequest();
     * 
     * 注意事项：
     * 1. 此方法依赖Laravel的请求上下文
     * 2. 在非HTTP请求环境(如命令行)中会返回null
     */
    static private function getCurrentRequest(): ?Request 
    {
        if (function_exists('request')) {
            return request();
        }
        return null;
    }
    // 私有方法：构建用户信息
    static private function buildUserInfo(?Request $request): string {
        if ($request && $request->user()) {
            $userObj = $request->user();
            $userInfo = "【-----------用户信息----------】".PHP_EOL;
            $userInfo .= "【用户ID】: ".$userObj->id.";".PHP_EOL; 
            $userInfo .= "【用户名称】: ".$userObj->name.";".PHP_EOL; 
            $requestParams = $request->all();
            $userInfo .= "【请求参数】: ".print_r($requestParams, true).PHP_EOL;
        } else {
            $userInfo = "【-----------用户信息----------】".PHP_EOL;
            $userInfo .= "【未获取到用户信息】".PHP_EOL;
        }
        return $userInfo;
    }

    // 私有方法：写入日志
    static private function writeLog(string $logPath, string $logKey, mixed $logInfo, string $userInfo, ?string $requestPath): void {
        $logMessage = "--------------------".str_repeat("-", 20).PHP_EOL;
        $logMessage .= "【".date('Y-m-d H:i:s')."】".PHP_EOL."【返回值：】".$userInfo."【请求路径】: $requestPath 【".$logKey."】: ".print_r($logInfo, true).PHP_EOL;
        $logMessage .= "--------------------".str_repeat("-", 20).PHP_EOL.PHP_EOL;
        file_put_contents($logPath, $logMessage, FILE_APPEND);
    }

    // 私有方法：处理异常
    static private function handleException(\Exception $e, string $methodName): void {
        $errorLogPath = storage_path("logs/a_error_log.log");
        file_put_contents($errorLogPath, "【".date('Y-m-d H:i:s')."】【{$methodName} 方法出错】: ". $e->getMessage(). PHP_EOL, FILE_APPEND);
    }

    // 私有方法：确保日志目录存在
    static private function ensureLogDir(string $logPath): void {
        if (!is_dir(dirname($logPath))) {
            mkdir(dirname($logPath), 0777, true);
        }
    }

    //保留兼容以前日志
    static public function add(string $logKey, mixed $logInfo): void {
        try {
            $request = self::getCurrentRequest();
            $day = date('Y-m-d');
            $logPath = storage_path("logs/a_runLog_".$day.".log");
            self::ensureLogDir($logPath);
            $userInfo = self::buildUserInfo($request);
            $requestPath = $request ? $request->getPathInfo() : 'N/A';
            self::writeLog($logPath, $logKey, $logInfo, $userInfo, $requestPath);
        } catch (\Exception $e) {
            self::handleException($e, 'add日志');
        }
    }
    
    // 中间件请求
    static public function RequestLog(string $logKey, mixed $logInfo,Request $Request=null): void {
        try {
            $request = self::getCurrentRequest();
            $day = date('Y-m-d');
            $logPath = storage_path("logs/a_runLog_Request".$day.".log");
            self::ensureLogDir($logPath);
            $userInfo = self::buildUserInfo($request);
            $requestPath = $request ? $request->getPathInfo() : 'N/A';
            self::writeLog($logPath, $logKey, $logInfo, $userInfo, $requestPath);
        } catch (\Exception $e) {
            self::handleException($e, 'RequestLog');
        }
    }

    // 错误日志
    static public function error(string $logKey, mixed $logInfo): void {
        self::writeNamedLog('a_error_', $logKey, $logInfo, 'addByName');
    }
    
    // 信息日志
    static public function info(mixed $logInfo): void {
        self::writeNamedLog('a_runLog_', 'debug', $logInfo, 'info');
    }

    // 私有方法：处理命名日志
    static private function writeNamedLog(string $prefix, string $logKey, mixed $logInfo, string $methodName): void {
        try {
            $request = self::getCurrentRequest();
            $logPath = storage_path("logs/{$prefix}{$logKey}.log");
            self::ensureLogDir($logPath);
            $userInfo = self::buildUserInfo($request);
            $requestPath = $request ? $request->getPathInfo() : 'N/A';
            self::writeLog($logPath, $logKey, $logInfo, $userInfo, $requestPath);
        } catch (\Exception $e) {
            self::handleException($e, $methodName);
        }
    }
}


