<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Command\Log; // 引入自定义日志类

class RequestLoggingMiddleware
{
    /**
     * 定义需要过滤的接口路径数组
     * @var array
     */
    private $filteredRoutes = [
        'auto-to-commentstatus',
        'simulate-login',
        'order-divide-record-dongjie',
        'uploads/*' // 新增图片路径过滤
    ];

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        
        // 检查请求的路径是否在过滤列表中
        foreach ($this->filteredRoutes as $route) {
            if ($request->is($route) || str_contains($request->path(), '.jpg')) {
                return $next($request);
            }
        }

        if (env('LOG_REQUESTS', false)) {
            // 记录请求信息
            $requestData = [
                'method' => $request->method(),
                'url' => $request->fullUrl(),
                'headers' => $request->headers->all(),
                'body' => $request->all(),
            ];

            // 截取请求参数值，每个最多 5000 字符
            $requestData = $this->truncateData($requestData);

            Log::RequestLog('请求参数', $requestData, $request); // 使用自定义日志类记录请求日志

            // 继续处理请求
            $response = $next($request);

            // 记录响应信息
            $responseData = [
                'status' => $response->getStatusCode(),
                'headers' => $response->headers->all(),
                'body' => $response->getContent(),
            ];

            // 截取响应参数值，每个最多 5000 字符
            $responseData = $this->truncateData($responseData);

            Log::RequestLog('响应', $responseData, $request); // 使用自定义日志类记录响应日志

            return $response;
        }

        return $next($request);
    }

    /**
     * 截取数据中的字符串值，每个最多 5000 字符
     *
     * @param array $data
     * @return array
     */
    private function truncateData(array $data)
    {
        foreach ($data as $key => $value) {
            if (is_string($value)) {
                $data[$key] = mb_substr($value, 0, 5000);
            } elseif (is_array($value)) {
                $data[$key] = $this->truncateData($value);
            }
        }
        return $data;
    }
}
