<?php

namespace App\Http\Controllers\Api;

use App\Command\Tools;
use App\Command\Log;
use App\Handlers\FileUploadHandler;
use App\Models\Merchant;
use App\Models\OrderDivideRecord;
use App\Models\OrderInfo;
use App\Models\Store;
use App\Models\StoreAdminUsers;
use App\Models\UserPermission;
use App\Models\PersonalAccessToken;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\OrderGoods;
use App\Models\Good;
use Illuminate\Support\Facades\DB;

class StoreAdminUsersController extends BaseController
{
/**
     * 用户登录方法
     *
     * @param Request $request HTTP请求对象
     * @return \Illuminate\Http\JsonResponse JSON响应
     */
    public function login(Request $request)
    {
        $user_type = $request->user_type ?? ''; //用户类型,1:商家端,2:核销员端,3:员工端【反正都是一个表单，去掉算了】
        $username = $request->username ?? '';
        $password = $request->password ?? '';

        $user = StoreAdminUsers::where(['username' => $username])->first();

        if (!$user) {
            return $this->JsonResponse('', '用户不存在', 201);
        }

        if (!Hash::check($password, $user->password)) {
            return $this->JsonResponse('', '账号或密码错误', 500);
        }

        // 生成token
        $accessToken = 'Bearer ' . $user->createToken('Access-token')->plainTextToken;

        // 记录登录日志
        Log::add('商家端用户--' . $user->id . '登录', ['token' => $accessToken]);

        return $this->JsonResponse([
            'Authorization' => $accessToken,
        ]);
    }

    /**
     * 用户注销方法
     *
     * @param Request $request HTTP请求对象
     * @return \Illuminate\Http\JsonResponse JSON响应
     */
    public function logout(Request $request)
    {
        $request->user()->tokens()->delete();
        PersonalAccessToken::where(['tokenable_id' => $request->user()->id])->delete();

        return $this->JsonResponse('');
    }

    /**
     * 获取用户信息方法
     *
     * @param Request $request HTTP请求对象
     * @return \Illuminate\Http\JsonResponse JSON响应
     */
    public function info(Request $request)
    {
        $muser = $request->user();

        $merchant_id = $muser->merchant_id;

        $total_revenue = $balance = $cashout = 0; //总收益，冻结中，已体现
       // $store_name = $merchant_name = $phone = $buycode = ''; //店铺名称，商户名称，手机号，购买码

        $balance01=$balance02=0;//临时使用

        $orderDivideRecord = [];
        if ($muser->role_id == 2) { //核销员
            //无需任何信息，直接返回
        } else {
            //查询冻结中+已提现，加起来就是总金额
            if ($muser->role_id == 1) { //商户
                //分佣记录[商户]
                $orderDivideRecord = OrderDivideRecord::where(['um_id' => $merchant_id])->select('divide_price', 'is_div', 'created_at')->orderBy('created_at', 'desc')->get();
           
                //订单状态为已支付，未分账的应该分佣金额
                $balance01=$this->getCommissionAmount(1,$merchant_id);
                Log::info('balance01-1:'.$balance01);
            } else {//员工
                //分佣记录[员工]
                $orderDivideRecord = OrderDivideRecord::where(['um_id' => $muser->member_id])->select('divide_price', 'is_div', 'created_at')->orderBy('created_at', 'desc')->get();
            
                //订单状态为已支付，未分账的应该分佣金额
                $balance01=$this->getCommissionAmount(2,$muser->id);
                Log::info('balance01-2:'.$balance01);
            }

            //分佣记录中，状态为0的，未绑卡的
            $balance02= $orderDivideRecord->where('is_div', 0)->sum('divide_price'); //冻结中
            Log::info('balance02:'.$balance02);
            
            //已提现==分佣记录中，状态为2的
            $cashout = number_format($orderDivideRecord->where('is_div', 1)->sum('divide_price'), 2); //已分账
            
            //冻结中=分佣记录中，状态为1的+订单状态为已支付，未分账的应该分佣金额
            $balance = number_format($balance01 + $balance02, 2); //冻结中
            $total_revenue = number_format($balance + $cashout, 2); //总佣金

            Log::info('balance:'.$balance);
        }

        $data=[
            'user_id' => $muser->id,
            'username' => $muser->username,
            'role_id' => $muser->role_id,
            'openid' => $muser->openid ?? '',
            'avatar' => $muser->avatar ? env('IMAGE_URL') . $muser->avatar : env('NO_AVATAR_IMAGE_URL'),
            'merchant_id' => $muser->merchant_id,
            'phone' => $muser->username,//商家用户表都没有手机号字段
            'buycode' => $muser->merchant ? $muser->merchant->buycode : '', // 增加空值判断
            'total_revenue' => $total_revenue,
            'balance' => $balance,
            'cashout' => $cashout,
            'store_name' => $muser->store ? $muser->store->title : $muser->merchant->name, // 增加空值判断
            'merchant_name' => $muser->merchant ? $muser->merchant->name : '', // 增加空值判断
        ];

        return $this->JsonResponse($data);
    }


    /**
     * 查询员工绑定用户列表
     *
     * @param Request $request HTTP请求对象
     * @return \Illuminate\Http\JsonResponse JSON响应
     * 
     * 返回数据结构:
     * {
     *     "total": 总记录数,
     *     "list": [
     *         {
     *             "id": 记录ID,
     *             "user_id": 用户ID,
     *             "username": 用户名,
     *             "avatar": 用户头像,
     *             "created_at": 绑定时间,
     *             "memo": 备注信息
     *         }
     *     ]
     * }
     * 
     * 使用示例:
     * GET /api/store-admin/employee-user-list
     * 
     * 注意事项:
     * 1. 仅员工角色可访问此接口
     * 2. 默认返回最近10条记录
     * 3. 支持分页参数page和limit
     * 4. 按创建时间倒序排列
     */
    public function getEmployeeUserList(Request $request)
    {
        $muser = $request->user();
        $page = $request->page ?? 1;
        $limit = $request->limit ?? 10;
        
        // 验证员工角色
        if ($muser->role_id != 3) {
            return $this->JsonResponse('', '无权限访问', 403);
        }
        
        $query = DB::table('store_employee_user_rec')
            ->leftJoin('users', 'users.id', '=', 'store_employee_user_rec.user_id')
            ->where('store_employee_user_rec.employee_id', $muser->id)
            ->orderBy('store_employee_user_rec.created_at', 'DESC');
            
        $total = $query->count();
        $list = $query->offset(($page - 1) * $limit)
            ->limit($limit)
            ->get([
                'store_employee_user_rec.user_id',
                'users.name',
                'users.phone',
                'users.avatar',
                'store_employee_user_rec.created_at',
                'store_employee_user_rec.memo'
            ])
            ->map(function ($item) {
                $item->avatar = $item->avatar ? env('IMAGE_URL') . $item->avatar : env('NO_AVATAR_IMAGE_URL');
                return $item;
            });
            
        return $this->JsonResponse([
            'total' => $total,
            'list' => $list
        ]);
    }


























    //------------------------------------------------------------------------------//
    //------------------------------以下是内部使用-----------------------------------//

    /**
     * 查询已支付订单，未分佣，属于对应用户的冻结金额
     * 
     * @param int $type 查询类型：1-商家冻结金额，2-员工冻结金额
     * @param int $obj_id 查询对象ID：商家ID或员工ID
     * @return float 冻结金额（保留2位小数）
     * @throws \Exception 当查询对象不存在时抛出异常
     * 注意事项：
     * 1. 该方法仅查询已支付(pay_type≠2)且未分佣(is_div=0)的订单
     * 2. 商家分佣计算：(商品数量*商品价格*商家佣金比例)/100
     * 3. 员工分佣计算：(商品数量*商品价格*商家佣金比例*员工佣金比例)/10000
     * 4. 混合支付(pay_type=2)会计算现金支付比例调整分佣金额
     * 5. 所有金额计算保留2位小数
     */
    public static function getCommissionAmount($type,  $obj_id)
    {
        $objUser = [];
        $orderObjList = [];
        if ($type == 1) {
            $objUser = Merchant::find($obj_id);

            //查询该对象所有已支付，未分佣的订单
            $orderObjList = DB::select('
                SELECT 
                    loi.id,
                    loi.pay_type,
                    log.goods_id,
                    loi.order_amount, 
                    log.goods_price, 
                    log.goods_number, 
                    -- 计算总的商品价格
                     COALESCE(log.goods_number*log.goods_price, 0) AS total_price,
                    -- 计算总的商户佣金（） 
                     
                    COALESCE( 
                        ROUND(((log.goods_price * lg.merchant_commission * 1.0) / 100) - ((log.goods_price * lg.merchant_commission * sau.employee_commission) / 10000), 2), 
                        ROUND(((log.goods_price * lg.merchant_commission * 1.0) / 100) , 2) 
                    ) AS total_merchant_commission,
                    -- 计算总的商户员工佣金 
                    COALESCE((log.goods_number * log.goods_price * lg.merchant_commission * sau.employee_commission) / 10000, 0) AS total_employee_commission
                FROM  
                    li_order_goods log 
                LEFT JOIN 
                    li_order_info loi ON loi.id = log.order_id 
                LEFT JOIN 
                    store_employee_user_rec seur ON seur.user_id = loi.user_id 
                LEFT JOIN 
                    store_admin_users sau ON sau.id = seur.employee_id
                LEFT JOIN 
                    li_goods lg ON lg.id = log.goods_id 
                WHERE 
                    loi.pay_type<>2 and loi.is_div=0 and (loi.order_status =3 || loi.order_status =4) and loi.merchant_id = ?; ', [$obj_id]);
                   
        // Log::add('zhu3',$obj_id); 
        // Log::add('zhu2',$orderObjList); 

        } else {
            $objUser = StoreAdminUsers::find($obj_id);

            //查询该对象所有已支付，未分佣的订单
            $orderObjList = DB::select('
               SELECT 
                    loi.id,
                    loi.pay_type,
                    log.goods_id,
                    loi.order_amount, 
                    log.goods_price, 
                    log.goods_number, 
                    -- 计算总的商品价格
                     COALESCE(log.goods_number*log.goods_price, 0) AS total_price,
                    -- 计算总的商户佣金 
                     COALESCE((log.goods_number*log.goods_price * lg.merchant_commission)/100, 0) AS total_merchant_commission,
                    -- 计算总的商户员工佣金 
                    COALESCE((log.goods_number * log.goods_price * lg.merchant_commission * sau.employee_commission) / 10000, 0) AS total_employee_commission
                FROM  
                    li_order_goods log 
                LEFT JOIN 
                    li_order_info loi ON loi.id = log.order_id 
                LEFT JOIN 
                    li_goods lg ON lg.id = log.goods_id 
                LEFT JOIN 
                    store_admin_users sau ON sau.id = ?
                WHERE 
                    loi.pay_type<>2 and loi.is_div=0 and (loi.order_status =3 || loi.order_status =4) and loi.user_id
                        in (
                            select user_id from  store_employee_user_rec where employee_id=?
                        ); ', [$obj_id, $obj_id]);
                        //Log::add('zhu3',$obj_id); 
        }

        //遍历列表，计算实际佣金
        $totalCommission = 0;

        //Log::add('zhu',$orderObjList);

        foreach ($orderObjList as $orderObj) {
            $cash_ratio = 1; //现金支付比例，默认100%
            //判断是否是混合支付
            if ($orderObj->pay_type == 3) {
                //计算现金支付比例
                $cash_ratio = number_format(($orderObj->order_amount / $orderObj->total_price) , 4); //保留4位小数
            }
            //应该分给商家的金额
            $merchant_divide_price = number_format($orderObj->total_merchant_commission * $cash_ratio, 4);
            
            //应该分给员工的金额
            $employee_divide_price =number_format($merchant_divide_price * ($objUser->employee_commission / 100), 4);

            Log::add('zhu7',$employee_divide_price);
            //判断计算商家还是员工
            if ($type == 1) {
                $totalCommission += $merchant_divide_price-$employee_divide_price;
            }else{
                $totalCommission += $employee_divide_price;
            }
        }
        Log::add('zhu8',$totalCommission);

        return $totalCommission;

    }
}
